#!/usr/bin/env bash

set -e

# Colors for output
RED='\033[0;31m'
GREEN='\033[0;32m'
BLUE='\033[0;34m'
YELLOW='\033[1;33m'
NC='\033[0m' # No Color

# Script directory
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
TEMPLATES_DIR="$(dirname "$SCRIPT_DIR")/templates"

echo -e "${BLUE}╔════════════════════════════════════════════╗${NC}"
echo -e "${BLUE}║   Terraform Module Generator - Skill #13  ║${NC}"
echo -e "${BLUE}╚════════════════════════════════════════════╝${NC}"
echo

# Function to prompt for selection
prompt_selection() {
    local prompt="$1"
    shift
    local options=("$@")

    echo -e "${YELLOW}$prompt${NC}"
    select opt in "${options[@]}"; do
        if [ -n "$opt" ]; then
            echo "$opt"
            return
        else
            echo -e "${RED}Invalid selection${NC}"
        fi
    done
}

# Cloud Provider Selection
echo -e "${GREEN}Step 1: Select Cloud Provider${NC}"
CLOUD_PROVIDER=$(prompt_selection "Choose cloud provider:" "AWS" "GCP" "Azure" "Cancel")

if [ "$CLOUD_PROVIDER" = "Cancel" ]; then
    echo "Cancelled."
    exit 0
fi

# Resource Type Selection
echo
echo -e "${GREEN}Step 2: Select Resource Type${NC}"
case $CLOUD_PROVIDER in
    "AWS")
        RESOURCE_TYPE=$(prompt_selection "Choose AWS resource:" "VPC" "EKS Cluster" "RDS Database" "Cancel")
        case $RESOURCE_TYPE in
            "VPC") TEMPLATE="aws-vpc" ;;
            "EKS Cluster") TEMPLATE="aws-eks" ;;
            "RDS Database") TEMPLATE="aws-rds" ;;
            *) echo "Cancelled."; exit 0 ;;
        esac
        ;;
    "GCP")
        RESOURCE_TYPE=$(prompt_selection "Choose GCP resource:" "GKE Cluster" "Cancel")
        case $RESOURCE_TYPE in
            "GKE Cluster") TEMPLATE="gcp-gke" ;;
            *) echo "Cancelled."; exit 0 ;;
        esac
        ;;
    "Azure")
        RESOURCE_TYPE=$(prompt_selection "Choose Azure resource:" "AKS Cluster" "Cancel")
        case $RESOURCE_TYPE in
            "AKS Cluster") TEMPLATE="azure-aks" ;;
            *) echo "Cancelled."; exit 0 ;;
        esac
        ;;
esac

# Destination Directory
echo
echo -e "${GREEN}Step 3: Specify Destination${NC}"
read -p "Enter destination directory [./terraform-modules]: " DEST_DIR
DEST_DIR=${DEST_DIR:-./terraform-modules}

# Module Name
echo
read -p "Enter module name [${TEMPLATE}]: " MODULE_NAME
MODULE_NAME=${MODULE_NAME:-$TEMPLATE}

# Full destination path
FULL_DEST="$DEST_DIR/$MODULE_NAME"

# Confirm
echo
echo -e "${BLUE}════════════════════════════════════════════${NC}"
echo "Cloud Provider: $CLOUD_PROVIDER"
echo "Resource Type:  $RESOURCE_TYPE"
echo "Template:       $TEMPLATE"
echo "Destination:    $FULL_DEST"
echo -e "${BLUE}════════════════════════════════════════════${NC}"
echo
read -p "Proceed? [Y/n]: " CONFIRM
CONFIRM=${CONFIRM:-Y}

if [[ ! "$CONFIRM" =~ ^[Yy]$ ]]; then
    echo "Cancelled."
    exit 0
fi

# Create destination directory
mkdir -p "$FULL_DEST"

# Copy template
echo
echo -e "${GREEN}Copying template...${NC}"
cp -r "$TEMPLATES_DIR/$TEMPLATE/"* "$FULL_DEST/"

# Success
echo
echo -e "${GREEN}✓ Module generated successfully!${NC}"
echo
echo "Next steps:"
echo "  1. cd $FULL_DEST"
echo "  2. cp terraform.tfvars.example terraform.tfvars"
echo "  3. Edit terraform.tfvars with your configuration"
echo "  4. terraform init"
echo "  5. terraform plan"
echo "  6. terraform apply"
echo
echo "Documentation: $FULL_DEST/README.md"
echo

# Offer to initialize
read -p "Initialize Terraform now? [y/N]: " INIT
if [[ "$INIT" =~ ^[Yy]$ ]]; then
    cd "$FULL_DEST"
    echo
    echo -e "${GREEN}Initializing Terraform...${NC}"
    terraform init
    echo
    echo -e "${GREEN}✓ Terraform initialized${NC}"
fi

echo
echo -e "${BLUE}Happy terraforming! 🚀${NC}"
