# Terraform Security Checklist - 2025

Production-grade security requirements for Terraform infrastructure.

## State Security

### Remote State

- [ ] Remote state configured (S3, GCS, Azure Storage)
- [ ] State encryption enabled
- [ ] State bucket versioning enabled
- [ ] State locking configured (DynamoDB, GCS locking, Azure lease)
- [ ] State bucket has restricted access (IAM policies)
- [ ] State bucket blocks public access
- [ ] Customer-managed KMS keys for encryption
- [ ] State access audit logging enabled (CloudTrail, Cloud Audit Logs)

### Local State (Development Only)

- [ ] .tfstate files in .gitignore
- [ ] No state files committed to version control
- [ ] Local state encrypted on disk

## Secrets Management

### No Hardcoded Secrets

- [ ] No passwords in .tf files
- [ ] No API keys in .tf files
- [ ] No access keys in .tf files
- [ ] No connection strings in .tf files
- [ ] .tfvars files with secrets in .gitignore
- [ ] terraform.tfvars.example provided (no real secrets)

### Secrets Storage

- [ ] Use AWS Secrets Manager / Azure Key Vault / GCP Secret Manager
- [ ] Use ephemeral resources for secrets (Terraform 1.10+)
- [ ] Secrets referenced via data sources, not hardcoded
- [ ] Sensitive outputs marked with `sensitive = true`
- [ ] Secrets rotation configured

### Example: Proper Secrets Usage

```hcl
# ✅ Good
data "aws_secretsmanager_secret_version" "db_password" {
  secret_id = "prod/db/password"
}

# ✅ Better (Terraform 1.10+)
ephemeral "aws_secretsmanager_secret_version" "db_password" {
  secret_id = "prod/db/password"
}

# ❌ Bad
variable "db_password" {
  default = "MyPassword123"
}
```

## Encryption

### Data at Rest

- [ ] S3 buckets encrypted (KMS)
- [ ] EBS volumes encrypted
- [ ] RDS databases encrypted
- [ ] DynamoDB tables encrypted
- [ ] EFS filesystems encrypted
- [ ] Snapshots encrypted
- [ ] Customer-managed KMS keys (not AWS-managed)
- [ ] KMS key rotation enabled

### Data in Transit

- [ ] TLS/SSL for all endpoints
- [ ] HTTPS-only for S3 buckets
- [ ] RDS connections use SSL
- [ ] VPN/Direct Connect for hybrid connectivity
- [ ] API Gateway uses TLS 1.2+

## IAM and Access Control

### Least Privilege

- [ ] IAM policies follow least privilege
- [ ] No wildcard (*) permissions unless necessary
- [ ] Resource-level permissions where possible
- [ ] No root account usage
- [ ] MFA required for sensitive operations
- [ ] Service accounts use IAM roles, not access keys

### RBAC

- [ ] Kubernetes RBAC configured
- [ ] Azure AD RBAC for AKS
- [ ] GCP Workload Identity for GKE
- [ ] EKS IRSA (IAM Roles for Service Accounts)

### Example: Least Privilege IAM

```hcl
# ✅ Good - Specific permissions
resource "aws_iam_policy" "app" {
  policy = jsonencode({
    Version = "2012-10-17"
    Statement = [{
      Effect = "Allow"
      Action = [
        "s3:GetObject",
        "s3:PutObject"
      ]
      Resource = "${aws_s3_bucket.main.arn}/*"
    }]
  })
}

# ❌ Bad - Overly permissive
resource "aws_iam_policy" "bad" {
  policy = jsonencode({
    Version = "2012-10-17"
    Statement = [{
      Effect   = "Allow"
      Action   = "*"
      Resource = "*"
    }]
  })
}
```

## Network Security

### VPC Configuration

- [ ] Private subnets for databases and applications
- [ ] Public subnets only for load balancers/bastion hosts
- [ ] NAT gateways for private subnet internet access
- [ ] VPC Flow Logs enabled
- [ ] Network ACLs configured
- [ ] Route tables reviewed

### Security Groups

- [ ] Least privilege ingress rules
- [ ] No 0.0.0.0/0 for sensitive ports (22, 3389, 3306, 5432)
- [ ] Egress rules restricted where possible
- [ ] Security group descriptions provided
- [ ] Security groups reference each other, not CIDR blocks

### Example: Secure Security Group

```hcl
# ✅ Good
resource "aws_security_group" "app" {
  name   = "app-sg"
  vpc_id = aws_vpc.main.id

  ingress {
    from_port       = 443
    to_port         = 443
    protocol        = "tcp"
    security_groups = [aws_security_group.alb.id]
    description     = "HTTPS from ALB"
  }
}

# ❌ Bad
resource "aws_security_group" "bad" {
  ingress {
    from_port   = 0
    to_port     = 65535
    protocol    = "tcp"
    cidr_blocks = ["0.0.0.0/0"]
  }
}
```

### Private Clusters

- [ ] EKS private endpoints enabled
- [ ] GKE private nodes configured
- [ ] AKS private cluster option enabled
- [ ] Database instances not publicly accessible

## Monitoring and Logging

### Audit Logging

- [ ] CloudTrail enabled (AWS)
- [ ] Cloud Audit Logs enabled (GCP)
- [ ] Activity Logs enabled (Azure)
- [ ] VPC Flow Logs enabled
- [ ] EKS control plane logging enabled
- [ ] RDS enhanced monitoring enabled
- [ ] Log retention configured (30+ days for production)

### Monitoring

- [ ] CloudWatch alarms for critical metrics
- [ ] Azure Monitor configured
- [ ] GCP Monitoring/Alerting configured
- [ ] Cost anomaly alerts enabled
- [ ] Security alerts configured

## Backup and Recovery

### Backup Configuration

- [ ] RDS automated backups enabled
- [ ] Backup retention >= 7 days (30+ for production)
- [ ] S3 versioning enabled
- [ ] EBS snapshot lifecycle policies
- [ ] Database point-in-time recovery enabled
- [ ] Cross-region backup replication for critical data

### Deletion Protection

- [ ] RDS deletion protection enabled
- [ ] S3 bucket MFA delete enabled
- [ ] EKS/GKE/AKS deletion protection
- [ ] Final snapshots configured

## Resource Configuration

### High Availability

- [ ] Multi-AZ deployments for databases
- [ ] Auto-scaling groups span multiple AZs
- [ ] Load balancers in multiple AZs
- [ ] EKS/GKE/AKS multi-zone node pools

### Updates and Patching

- [ ] Auto-update enabled for managed services
- [ ] AMI update strategy defined
- [ ] Kubernetes version upgrade plan
- [ ] Database maintenance windows configured

## Code Security

### Static Analysis

- [ ] tfsec scans passing
- [ ] Checkov scans passing
- [ ] Trivy scans passing
- [ ] No high/critical findings unresolved

### Version Control

- [ ] All Terraform code in version control
- [ ] Branch protection enabled
- [ ] Pull request reviews required
- [ ] No secrets in git history
- [ ] .gitignore includes sensitive files

### Example: .gitignore

```
# Terraform files
.terraform/
*.tfstate
*.tfstate.*
crash.log
*.tfvars
!terraform.tfvars.example

# Secrets
.env
*.pem
*.key
secrets/
```

## Provider Configuration

### Version Pinning

- [ ] Terraform version specified (>= x.y)
- [ ] Provider versions pinned (~> x.y)
- [ ] Module versions specified
- [ ] No experimental features in production

### Provider Authentication

- [ ] Use OIDC for CI/CD (not long-lived credentials)
- [ ] AWS: Use IAM roles, not access keys
- [ ] GCP: Use Workload Identity
- [ ] Azure: Use Managed Identity
- [ ] No credentials in provider blocks

```hcl
# ✅ Good - Use environment variables or IAM roles
provider "aws" {
  region = "us-east-1"
  # Credentials from environment or instance profile
}

# ❌ Bad - Hardcoded credentials
provider "aws" {
  access_key = "AKIAIOSFODNN7EXAMPLE"
  secret_key = "wJalrXUtnFEMI/K7MDENG/bPxRfiCYEXAMPLEKEY"
}
```

## Compliance

### Industry Standards

- [ ] CIS benchmarks applied
- [ ] SOC 2 requirements met
- [ ] HIPAA compliance (if applicable)
- [ ] PCI DSS compliance (if applicable)
- [ ] GDPR compliance (if applicable)

### Tagging

- [ ] All resources tagged
- [ ] Required tags: Environment, ManagedBy, CostCenter, Owner
- [ ] Tag compliance policy enforced
- [ ] Cost allocation tags configured

## Kubernetes Security (EKS/GKE/AKS)

### Cluster Security

- [ ] Network policies enabled
- [ ] Pod security policies/standards configured
- [ ] RBAC enabled
- [ ] Secrets encrypted at rest
- [ ] Image scanning enabled
- [ ] Private container registry
- [ ] No privileged containers in production

### Workload Security

- [ ] Service mesh configured (Istio, Linkerd)
- [ ] mTLS between services
- [ ] Network policies restrict pod communication
- [ ] Resource quotas configured
- [ ] Limit ranges configured

## CI/CD Security

### Pipeline Security

- [ ] Terraform plan reviewed before apply
- [ ] Manual approval for production
- [ ] No auto-apply in production
- [ ] Secrets stored in CI/CD secret manager
- [ ] Pipeline runs with minimal permissions
- [ ] Audit log of all deployments

### Testing

- [ ] Terraform validate in pipeline
- [ ] terraform fmt check in pipeline
- [ ] Security scans in pipeline
- [ ] Policy as code (OPA, Sentinel)
- [ ] Cost estimation in pipeline

## Incident Response

### Preparedness

- [ ] Incident response plan documented
- [ ] Terraform state backup procedure
- [ ] Rollback procedure documented
- [ ] Emergency access procedure
- [ ] Contact information documented

### Recovery

- [ ] Disaster recovery plan tested
- [ ] State recovery procedure documented
- [ ] Infrastructure reproducible from code
- [ ] Regular backup testing

## Quick Security Scan

Run these commands before deployment:

```bash
# Format and validate
terraform fmt -check -recursive
terraform validate

# Security scanning
tfsec . --minimum-severity MEDIUM
checkov -d . --framework terraform
trivy config .

# Cost estimation
infracost breakdown --path .

# Plan review
terraform plan -out=tfplan
terraform show -json tfplan | jq
```

## Summary

### Critical Requirements

1. **No secrets in code** - Use Secrets Manager/Key Vault
2. **Encryption everywhere** - At rest and in transit
3. **Least privilege IAM** - Minimal required permissions
4. **Remote state encrypted** - With locking enabled
5. **Multi-AZ/zone** - For production workloads
6. **Audit logging** - CloudTrail/Cloud Audit Logs
7. **Security scanning** - tfsec/Checkov in CI/CD
8. **Version pinning** - Terraform and providers

### Before Production Deployment

- [ ] All checklist items reviewed
- [ ] Security scan results reviewed
- [ ] Peer review completed
- [ ] Backup and recovery tested
- [ ] Monitoring and alerts configured
- [ ] Incident response plan ready
