# Terraform Module Writer - Quick Start

Get started with Terraform modules in 1 minute.

## Installation

```bash
# Install Terraform
brew install terraform  # macOS
# or
wget https://releases.hashicorp.com/terraform/1.6.0/terraform_1.6.0_linux_amd64.zip

# Verify installation
terraform version
```

## Create Your First Module (60 seconds)

### 1. Choose a Template

```bash
cd terraform-module-writer
./scripts/generate-module.sh
```

Select:
- Cloud provider (AWS/GCP/Azure)
- Resource type (VPC/EKS/RDS/GKE/AKS)

### 2. Customize Variables

```bash
cd templates/aws-vpc
cp terraform.tfvars.example terraform.tfvars
# Edit terraform.tfvars with your values
```

### 3. Deploy

```bash
# Initialize
terraform init

# Review plan
terraform plan

# Deploy
terraform apply
```

## Basic Module Structure

Every module needs these 4 files:

```
my-module/
├── main.tf         # Resources
├── variables.tf    # Inputs
├── outputs.tf      # Outputs
└── versions.tf     # Provider versions
```

## Minimal Working Example

### main.tf
```hcl
resource "aws_vpc" "main" {
  cidr_block = var.vpc_cidr

  tags = {
    Name = var.vpc_name
  }
}
```

### variables.tf
```hcl
variable "vpc_name" {
  type = string
}

variable "vpc_cidr" {
  type = string
}
```

### outputs.tf
```hcl
output "vpc_id" {
  value = aws_vpc.main.id
}
```

### versions.tf
```hcl
terraform {
  required_version = ">= 1.6"

  required_providers {
    aws = {
      source  = "hashicorp/aws"
      version = "~> 5.0"
    }
  }
}
```

## Common Commands

```bash
terraform init          # Initialize directory
terraform fmt           # Format code
terraform validate      # Validate configuration
terraform plan          # Preview changes
terraform apply         # Apply changes
terraform destroy       # Destroy infrastructure
terraform output        # Show outputs
```

## Using a Module

```hcl
module "vpc" {
  source = "./modules/aws-vpc"

  vpc_name = "production-vpc"
  vpc_cidr = "10.0.0.0/16"
}

# Reference module outputs
resource "aws_subnet" "app" {
  vpc_id = module.vpc.vpc_id
  # ...
}
```

## Remote State (Production)

```hcl
terraform {
  backend "s3" {
    bucket         = "my-terraform-state"
    key            = "production/terraform.tfstate"
    region         = "us-east-1"
    encrypt        = true
    dynamodb_table = "terraform-state-lock"
  }
}
```

## Security Essentials

1. **Never commit secrets**
   ```bash
   # Add to .gitignore
   *.tfstate
   *.tfvars
   .terraform/
   ```

2. **Mark sensitive outputs**
   ```hcl
   output "password" {
     value     = random_password.db.result
     sensitive = true
   }
   ```

3. **Use variable validation**
   ```hcl
   variable "environment" {
     type = string
     validation {
       condition     = contains(["dev", "prod"], var.environment)
       error_message = "Must be dev or prod."
     }
   }
   ```

## Troubleshooting

### Error: Backend not initialized
```bash
terraform init
```

### Error: Resource already exists
```bash
terraform import aws_vpc.main vpc-12345678
```

### Error: State locked
```bash
terraform force-unlock <LOCK_ID>
```

## Next Steps

1. **Explore templates** - Check `templates/` directory
2. **Read best practices** - See `resources/best-practices.md`
3. **Security review** - See `resources/security-checklist.md`
4. **Validate** - Run `./scripts/validate-terraform.sh`

## Resources

- **Official Docs**: https://www.terraform.io/docs
- **Module Registry**: https://registry.terraform.io
- **Best Practices**: `resources/best-practices.md`
- **Security**: `resources/security-checklist.md`

## Quick Reference Card

| Task | Command |
|------|---------|
| Initialize | `terraform init` |
| Format | `terraform fmt` |
| Validate | `terraform validate` |
| Plan | `terraform plan` |
| Apply | `terraform apply` |
| Destroy | `terraform destroy` |
| Show state | `terraform show` |
| List resources | `terraform state list` |
| Import | `terraform import <resource> <id>` |
| Unlock | `terraform force-unlock <id>` |

You're ready to build infrastructure! 🚀
