# Terraform Best Practices - 2025 Standards

Comprehensive guide to production-ready Terraform modules based on 2025 industry standards.

## Module Structure

### Standard Module Layout

```
module-name/
├── README.md              # Module documentation
├── main.tf                # Primary resource definitions
├── variables.tf           # Input variables
├── outputs.tf             # Output values
├── versions.tf            # Provider version constraints
├── terraform.tfvars.example  # Example variable values
├── examples/
│   └── basic/
│       ├── main.tf        # Working example
│       └── README.md      # Example documentation
└── tests/                 # Optional: Terratest or similar
    └── module_test.go
```

### File Organization

- **main.tf**: Core resource definitions, grouped logically
- **variables.tf**: All input variables with types and validation
- **outputs.tf**: All output values with descriptions
- **versions.tf**: Terraform and provider version constraints
- **locals.tf**: (Optional) Local values and computations
- **data.tf**: (Optional) Data source definitions

## Remote State Management

### S3 Backend (AWS)

```hcl
terraform {
  backend "s3" {
    bucket         = "my-terraform-state"
    key            = "production/terraform.tfstate"
    region         = "us-east-1"
    encrypt        = true                    # REQUIRED
    dynamodb_table = "terraform-state-lock"  # REQUIRED for locking
    kms_key_id     = "arn:aws:kms:us-east-1:123456789012:key/..."
  }
}
```

### Creating Backend Infrastructure

```hcl
# S3 bucket for state
resource "aws_s3_bucket" "terraform_state" {
  bucket = "my-terraform-state"
}

# Enable versioning
resource "aws_s3_bucket_versioning" "terraform_state" {
  bucket = aws_s3_bucket.terraform_state.id
  versioning_configuration {
    status = "Enabled"
  }
}

# Enable encryption
resource "aws_s3_bucket_server_side_encryption_configuration" "terraform_state" {
  bucket = aws_s3_bucket.terraform_state.id

  rule {
    apply_server_side_encryption_by_default {
      sse_algorithm     = "aws:kms"
      kms_master_key_id = aws_kms_key.terraform_state.arn
    }
  }
}

# Block public access
resource "aws_s3_bucket_public_access_block" "terraform_state" {
  bucket = aws_s3_bucket.terraform_state.id

  block_public_acls       = true
  block_public_policy     = true
  ignore_public_acls      = true
  restrict_public_buckets = true
}

# DynamoDB table for state locking
resource "aws_dynamodb_table" "terraform_locks" {
  name         = "terraform-state-lock"
  billing_mode = "PAY_PER_REQUEST"
  hash_key     = "LockID"

  attribute {
    name = "LockID"
    type = "S"
  }
}
```

### GCS Backend (GCP)

```hcl
terraform {
  backend "gcs" {
    bucket = "my-terraform-state"
    prefix = "production"
    encryption_key = "..." # Customer-managed encryption key
  }
}
```

### Azure Backend

```hcl
terraform {
  backend "azurerm" {
    resource_group_name  = "terraform-state-rg"
    storage_account_name = "tfstate"
    container_name       = "tfstate"
    key                  = "production.terraform.tfstate"
    use_azuread_auth     = true
  }
}
```

## Variable Validation

### Type Constraints

```hcl
variable "environment" {
  description = "Environment name"
  type        = string

  validation {
    condition     = contains(["dev", "staging", "production"], var.environment)
    error_message = "Environment must be dev, staging, or production."
  }
}

variable "vpc_cidr" {
  description = "VPC CIDR block"
  type        = string

  validation {
    condition     = can(cidrhost(var.vpc_cidr, 0))
    error_message = "VPC CIDR must be a valid IPv4 CIDR block."
  }
}

variable "instance_count" {
  description = "Number of instances"
  type        = number

  validation {
    condition     = var.instance_count > 0 && var.instance_count <= 100
    error_message = "Instance count must be between 1 and 100."
  }
}

variable "tags" {
  description = "Resource tags"
  type        = map(string)
  default     = {}

  validation {
    condition     = contains(keys(var.tags), "Environment")
    error_message = "Tags must include 'Environment' key."
  }
}
```

### Complex Type Validation

```hcl
variable "node_groups" {
  description = "EKS node group configurations"
  type = map(object({
    instance_types   = list(string)
    desired_capacity = number
    min_capacity     = number
    max_capacity     = number
  }))

  validation {
    condition = alltrue([
      for k, v in var.node_groups :
      v.min_capacity <= v.desired_capacity && v.desired_capacity <= v.max_capacity
    ])
    error_message = "For each node group: min <= desired <= max capacity."
  }
}
```

## Output Organization

### Grouping Outputs

```hcl
# outputs.tf

# VPC Outputs
output "vpc_id" {
  description = "The ID of the VPC"
  value       = aws_vpc.main.id
}

output "vpc_cidr_block" {
  description = "The CIDR block of the VPC"
  value       = aws_vpc.main.cidr_block
}

# Subnet Outputs
output "public_subnet_ids" {
  description = "List of IDs of public subnets"
  value       = aws_subnet.public[*].id
}

output "private_subnet_ids" {
  description = "List of IDs of private subnets"
  value       = aws_subnet.private[*].id
}

# Sensitive Outputs
output "database_password" {
  description = "Database administrator password"
  value       = random_password.db_password.result
  sensitive   = true  # REQUIRED for secrets
}
```

### Output Preconditions (Terraform 1.2+)

```hcl
output "vpc_id" {
  description = "The ID of the VPC"
  value       = aws_vpc.main.id

  precondition {
    condition     = aws_vpc.main.enable_dns_hostnames == true
    error_message = "VPC must have DNS hostnames enabled."
  }
}
```

## Security Best Practices

### 1. No Hardcoded Secrets

**❌ Bad:**
```hcl
resource "aws_db_instance" "main" {
  username = "admin"
  password = "MyPassword123"  # NEVER do this!
}
```

**✅ Good:**
```hcl
# Use AWS Secrets Manager
data "aws_secretsmanager_secret_version" "db_password" {
  secret_id = "prod/db/password"
}

resource "aws_db_instance" "main" {
  username = "admin"
  password = data.aws_secretsmanager_secret_version.db_password.secret_string
}
```

### 2. Ephemeral Resources (Terraform 1.10+)

```hcl
# Secrets retrieved at runtime, not stored in state
ephemeral "aws_secretsmanager_secret_version" "db_password" {
  secret_id = "prod/db/password"
}

resource "aws_db_instance" "main" {
  username = "admin"
  password = ephemeral.aws_secretsmanager_secret_version.db_password.secret_string
}
```

### 3. Encryption at Rest

```hcl
# S3 Bucket
resource "aws_s3_bucket_server_side_encryption_configuration" "main" {
  bucket = aws_s3_bucket.main.id

  rule {
    apply_server_side_encryption_by_default {
      sse_algorithm     = "aws:kms"
      kms_master_key_id = aws_kms_key.main.arn
    }
  }
}

# RDS Instance
resource "aws_db_instance" "main" {
  storage_encrypted = true
  kms_key_id        = aws_kms_key.main.arn
}

# EBS Volume
resource "aws_ebs_volume" "main" {
  encrypted  = true
  kms_key_id = aws_kms_key.main.arn
}
```

### 4. Least Privilege IAM

```hcl
# Specific permissions, not wildcards
resource "aws_iam_policy" "app" {
  name = "app-policy"

  policy = jsonencode({
    Version = "2012-10-17"
    Statement = [
      {
        Effect = "Allow"
        Action = [
          "s3:GetObject",
          "s3:PutObject"
        ]
        Resource = "${aws_s3_bucket.main.arn}/*"
      }
    ]
  })
}
```

## Module Versioning

### Semantic Versioning

```hcl
module "vpc" {
  source  = "terraform-aws-modules/vpc/aws"
  version = "~> 5.0"  # Allow 5.x updates, not 6.0
}
```

### Version Constraints

- `~> 5.0` - Allow 5.x (5.0, 5.1, 5.2, etc.), not 6.0
- `>= 5.0, < 6.0` - Explicit range
- `= 5.0.0` - Exact version (not recommended)

## Naming Conventions

### Resources

Use `snake_case` for all Terraform identifiers:

```hcl
resource "aws_vpc" "main" {}
resource "aws_subnet" "public" {}
resource "aws_security_group" "web_server" {}
```

### Variables and Outputs

```hcl
variable "vpc_cidr_block" {}
variable "enable_nat_gateway" {}

output "vpc_id" {}
output "private_subnet_ids" {}
```

## Common Patterns

### Conditional Resources

```hcl
resource "aws_nat_gateway" "main" {
  count = var.enable_nat_gateway ? length(var.availability_zones) : 0

  allocation_id = aws_eip.nat[count.index].id
  subnet_id     = aws_subnet.public[count.index].id
}
```

### Dynamic Blocks

```hcl
resource "aws_security_group" "main" {
  name   = "web-sg"
  vpc_id = aws_vpc.main.id

  dynamic "ingress" {
    for_each = var.ingress_rules
    content {
      from_port   = ingress.value.from_port
      to_port     = ingress.value.to_port
      protocol    = ingress.value.protocol
      cidr_blocks = ingress.value.cidr_blocks
    }
  }
}
```

### For Expressions

```hcl
locals {
  subnet_ids = [for subnet in aws_subnet.private : subnet.id]

  subnet_map = {
    for az, subnet in aws_subnet.private :
    az => subnet.id
  }
}
```

### Locals for Computed Values

```hcl
locals {
  common_tags = merge(
    var.tags,
    {
      ManagedBy   = "terraform"
      Environment = var.environment
    }
  )

  availability_zones = slice(data.aws_availability_zones.available.names, 0, var.az_count)
}
```

## Testing and Validation

### Pre-deployment Validation

```bash
# Format check
terraform fmt -check -recursive

# Validation
terraform validate

# Plan review
terraform plan -out=tfplan

# Security scanning
tfsec .
checkov -d .
```

### CI/CD Integration

```yaml
# GitHub Actions example
- name: Terraform Format
  run: terraform fmt -check -recursive

- name: Terraform Validate
  run: |
    terraform init -backend=false
    terraform validate

- name: tfsec
  uses: aquasecurity/tfsec-action@v1.0.0
```

## Performance Optimization

### Use Data Sources Efficiently

```hcl
# Cache AMI lookups
data "aws_ami" "ubuntu" {
  most_recent = true
  owners      = ["099720109477"]

  filter {
    name   = "name"
    values = ["ubuntu/images/hvm-ssd/ubuntu-jammy-22.04-amd64-server-*"]
  }
}
```

### Targeted Operations

```bash
# Only refresh specific resources
terraform plan -target=module.vpc

# Parallelism control
terraform apply -parallelism=20
```

## Documentation Standards

### README.md Template

```markdown
# Module Name

Brief description of what the module does.

## Features

- Feature 1
- Feature 2

## Usage

\`\`\`hcl
module "example" {
  source = "./module"
  # ...
}
\`\`\`

## Requirements

| Name | Version |
|------|---------|
| terraform | >= 1.6 |
| aws | ~> 5.0 |

## Inputs

| Name | Description | Type | Default | Required |
|------|-------------|------|---------|----------|
| ... | ... | ... | ... | ... |

## Outputs

| Name | Description |
|------|-------------|
| ... | ... |
```

## Summary Checklist

- [ ] Use remote state with encryption
- [ ] Enable state locking (DynamoDB for AWS)
- [ ] Add variable validation
- [ ] Mark sensitive outputs
- [ ] No hardcoded secrets
- [ ] Encryption enabled for data stores
- [ ] Use semantic versioning for modules
- [ ] Follow snake_case naming
- [ ] Include comprehensive documentation
- [ ] Add working examples
- [ ] Test with terraform validate
- [ ] Format with terraform fmt
- [ ] Scan with tfsec/checkov
