#!/usr/bin/env bash

# Prometheus Configuration Test Suite
# Tests all templates and generated configurations
# Last Updated: January 2025

set -euo pipefail

# Colors
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

print_info() { echo -e "${BLUE}ℹ${NC} $1"; }
print_success() { echo -e "${GREEN}✓${NC} $1"; }
print_warning() { echo -e "${YELLOW}⚠${NC} $1"; }
print_error() { echo -e "${RED}✗${NC} $1"; }

# Test counters
TESTS_RUN=0
TESTS_PASSED=0
TESTS_FAILED=0

# Script directory
SCRIPT_DIR="$(cd "$(dirname "${BASH_SOURCE[0]}")" && pwd)"
PROJECT_DIR="$(dirname "$SCRIPT_DIR")"
TEMPLATES_DIR="$PROJECT_DIR/templates"

print_header() {
    echo ""
    echo "═══════════════════════════════════════════════════════════════"
    echo "   Prometheus Configuration Test Suite"
    echo "═══════════════════════════════════════════════════════════════"
    echo ""
}

# Check if promtool is available
check_promtool() {
    if ! command -v promtool &> /dev/null; then
        print_warning "promtool not found, using Docker"
        PROMTOOL="docker run --rm -v $PWD:/config prom/prometheus:latest promtool"
    else
        PROMTOOL="promtool"
        print_success "Using local promtool: $(promtool --version | head -1)"
    fi
}

# Test a Prometheus configuration file
test_config() {
    local config_file="$1"
    local test_name="$(basename "$config_file")"

    TESTS_RUN=$((TESTS_RUN + 1))

    print_info "Testing: $test_name"

    if $PROMTOOL check config "$config_file" &> /dev/null; then
        TESTS_PASSED=$((TESTS_PASSED + 1))
        print_success "PASS: $test_name"
        return 0
    else
        TESTS_FAILED=$((TESTS_FAILED + 1))
        print_error "FAIL: $test_name"
        $PROMTOOL check config "$config_file" || true
        return 1
    fi
}

# Test alert rules
test_rules() {
    local rules_file="$1"
    local test_name="$(basename "$rules_file")"

    TESTS_RUN=$((TESTS_RUN + 1))

    print_info "Testing: $test_name"

    if $PROMTOOL check rules "$rules_file" &> /dev/null; then
        TESTS_PASSED=$((TESTS_PASSED + 1))
        print_success "PASS: $test_name"
        return 0
    else
        TESTS_FAILED=$((TESTS_FAILED + 1))
        print_error "FAIL: $test_name"
        $PROMTOOL check rules "$rules_file" || true
        return 1
    fi
}

# Test all Prometheus configurations
test_all_configs() {
    echo ""
    print_info "Testing Prometheus configurations..."
    echo ""

    # Test kubernetes monitoring
    if [ -f "$TEMPLATES_DIR/kubernetes-monitoring/prometheus.yml" ]; then
        test_config "$TEMPLATES_DIR/kubernetes-monitoring/prometheus.yml"
    fi

    # Test docker monitoring
    if [ -f "$TEMPLATES_DIR/docker-monitoring/prometheus.yml" ]; then
        test_config "$TEMPLATES_DIR/docker-monitoring/prometheus.yml"
    fi

    # Test node exporter
    if [ -f "$TEMPLATES_DIR/node-exporter/prometheus.yml" ]; then
        test_config "$TEMPLATES_DIR/node-exporter/prometheus.yml"
    fi
}

# Test all alert rules
test_all_alert_rules() {
    echo ""
    print_info "Testing alert rules..."
    echo ""

    for rules_file in "$TEMPLATES_DIR/alerting-rules"/*.yml; do
        if [ -f "$rules_file" ]; then
            test_rules "$rules_file"
        fi
    done
}

# Test all recording rules
test_all_recording_rules() {
    echo ""
    print_info "Testing recording rules..."
    echo ""

    for rules_file in "$TEMPLATES_DIR/recording-rules"/*.yml; do
        if [ -f "$rules_file" ]; then
            test_rules "$rules_file"
        fi
    done
}

# Test YAML syntax (without promtool)
test_yaml_syntax() {
    echo ""
    print_info "Testing YAML syntax..."
    echo ""

    local yaml_files=(
        "$TEMPLATES_DIR/service-discovery"/*.yml
    )

    for yaml_file in "${yaml_files[@]}"; do
        if [ -f "$yaml_file" ]; then
            TESTS_RUN=$((TESTS_RUN + 1))
            local test_name="$(basename "$yaml_file")"

            if python3 -c "import yaml; yaml.safe_load(open('$yaml_file'))" 2>/dev/null; then
                TESTS_PASSED=$((TESTS_PASSED + 1))
                print_success "PASS: $test_name (YAML syntax)"
            else
                TESTS_FAILED=$((TESTS_FAILED + 1))
                print_error "FAIL: $test_name (YAML syntax)"
            fi
        fi
    done
}

# Test scripts are executable
test_scripts() {
    echo ""
    print_info "Testing scripts..."
    echo ""

    local scripts=(
        "$PROJECT_DIR/scripts/generate-prometheus-config.sh"
        "$PROJECT_DIR/scripts/validate-prometheus-config.sh"
    )

    for script in "${scripts[@]}"; do
        TESTS_RUN=$((TESTS_RUN + 1))
        local test_name="$(basename "$script")"

        if [ -x "$script" ]; then
            TESTS_PASSED=$((TESTS_PASSED + 1))
            print_success "PASS: $test_name (executable)"
        else
            TESTS_FAILED=$((TESTS_FAILED + 1))
            print_error "FAIL: $test_name (not executable)"
        fi
    done
}

# Test documentation exists
test_documentation() {
    echo ""
    print_info "Testing documentation..."
    echo ""

    local docs=(
        "$PROJECT_DIR/README.md"
        "$PROJECT_DIR/skill.md"
        "$PROJECT_DIR/resources/best-practices.md"
    )

    for doc in "${docs[@]}"; do
        TESTS_RUN=$((TESTS_RUN + 1))
        local test_name="$(basename "$doc")"

        if [ -f "$doc" ] && [ -s "$doc" ]; then
            TESTS_PASSED=$((TESTS_PASSED + 1))
            print_success "PASS: $test_name (exists and not empty)"
        else
            TESTS_FAILED=$((TESTS_FAILED + 1))
            print_error "FAIL: $test_name (missing or empty)"
        fi
    done
}

# Print test summary
print_summary() {
    echo ""
    echo "═══════════════════════════════════════════════════════════════"
    echo "   Test Summary"
    echo "═══════════════════════════════════════════════════════════════"
    echo ""
    echo "Tests Run:    $TESTS_RUN"
    echo -e "Tests Passed: ${GREEN}$TESTS_PASSED${NC}"
    echo -e "Tests Failed: ${RED}$TESTS_FAILED${NC}"
    echo ""

    if [ $TESTS_FAILED -eq 0 ]; then
        print_success "All tests passed!"
        return 0
    else
        print_error "Some tests failed"
        return 1
    fi
}

# Main execution
main() {
    print_header

    # Check for promtool
    check_promtool

    # Run all tests
    test_all_configs
    test_all_alert_rules
    test_all_recording_rules
    test_yaml_syntax
    test_scripts
    test_documentation

    # Print summary
    print_summary
}

# Run main
main "$@"
