# Nginx Best Practices Guide - 2025 Standards

Comprehensive guide for securing, optimizing, and maintaining production Nginx configurations based on current industry standards and security recommendations.

## Table of Contents

1. [Security Best Practices](#security-best-practices)
2. [SSL/TLS Configuration](#ssltls-configuration)
3. [Performance Optimization](#performance-optimization)
4. [Rate Limiting and DDoS Protection](#rate-limiting-and-ddos-protection)
5. [Logging and Monitoring](#logging-and-monitoring)
6. [High Availability](#high-availability)
7. [Common Pitfalls](#common-pitfalls)
8. [Maintenance and Updates](#maintenance-and-updates)

## Security Best Practices

### 1. Hide Nginx Version

**Why**: Exposing version information helps attackers identify vulnerabilities.

```nginx
http {
    server_tokens off;
}
```

For complete removal including error pages, use the `headers-more` module:
```nginx
more_clear_headers Server;
```

### 2. Restrict HTTP Methods

Only allow methods your application needs:

```nginx
location / {
    if ($request_method !~ ^(GET|HEAD|POST|PUT|DELETE|OPTIONS)$) {
        return 405;
    }
}
```

### 3. Implement Security Headers

Essential headers for 2025:

```nginx
# HSTS (HTTP Strict Transport Security)
add_header Strict-Transport-Security "max-age=31536000; includeSubDomains; preload" always;

# Prevent clickjacking
add_header X-Frame-Options "SAMEORIGIN" always;

# Prevent MIME sniffing
add_header X-Content-Type-Options "nosniff" always;

# XSS Protection (legacy browsers)
add_header X-XSS-Protection "1; mode=block" always;

# Content Security Policy
add_header Content-Security-Policy "default-src 'self'; script-src 'self' 'unsafe-inline'; style-src 'self' 'unsafe-inline';" always;

# Referrer Policy
add_header Referrer-Policy "strict-origin-when-cross-origin" always;

# Permissions Policy (formerly Feature-Policy)
add_header Permissions-Policy "geolocation=(), microphone=(), camera=()" always;
```

**Testing**: Use https://securityheaders.com/ to verify your headers configuration.

### 4. Buffer Overflow Protection

Protect against buffer overflow attacks:

```nginx
client_body_buffer_size 128k;
client_max_body_size 10m;
client_header_buffer_size 1k;
large_client_header_buffers 4 16k;
```

Adjust `client_max_body_size` based on your application needs.

### 5. Deny Access to Hidden Files

Prevent access to sensitive files:

```nginx
location ~ /\. {
    deny all;
    access_log off;
    log_not_found off;
}

# Specific exceptions (like Let's Encrypt)
location ~ /\.well-known {
    allow all;
}
```

### 6. Disable Directory Listing

```nginx
autoindex off;
```

### 7. Access Control

Implement IP whitelisting for sensitive endpoints:

```nginx
location /admin {
    allow 10.0.0.0/8;
    allow 192.168.0.0/16;
    deny all;
}
```

### 8. Disable Unused Modules

Compile Nginx with only required modules to reduce attack surface:

```bash
# Check current modules
nginx -V 2>&1 | grep -o with-http_[a-z_]* | sort | uniq

# Rebuild without unused modules
./configure --without-http_autoindex_module --without-http_ssi_module
```

## SSL/TLS Configuration

### Modern TLS Configuration (2025)

**Minimum Requirements:**
- TLS 1.2 and TLS 1.3 only
- Strong cipher suites
- OCSP stapling
- HTTP/2 support

### Complete SSL Configuration

```nginx
server {
    listen 443 ssl http2;
    listen [::]:443 ssl http2;

    # Certificates
    ssl_certificate /etc/ssl/certs/example.com-fullchain.crt;
    ssl_certificate_key /etc/ssl/private/example.com.key;

    # Protocols (TLS 1.2 and 1.3 only)
    ssl_protocols TLSv1.2 TLSv1.3;

    # Cipher suites (TLS 1.2 - TLS 1.3 handles its own)
    ssl_ciphers 'ECDHE-ECDSA-AES128-GCM-SHA256:ECDHE-RSA-AES128-GCM-SHA256:ECDHE-ECDSA-AES256-GCM-SHA384:ECDHE-RSA-AES256-GCM-SHA384:ECDHE-ECDSA-CHACHA20-POLY1305:ECDHE-RSA-CHACHA20-POLY1305:DHE-RSA-AES128-GCM-SHA256:DHE-RSA-AES256-GCM-SHA384';

    # Let client choose cipher for TLS 1.3
    ssl_prefer_server_ciphers off;

    # DH parameters (4096-bit for maximum security)
    ssl_dhparam /etc/ssl/certs/dhparam.pem;

    # Session settings
    ssl_session_timeout 1d;
    ssl_session_cache shared:SSL:50m;
    ssl_session_tickets off;  # Disable for Perfect Forward Secrecy

    # OCSP Stapling
    ssl_stapling on;
    ssl_stapling_verify on;
    ssl_trusted_certificate /etc/ssl/certs/ca-bundle.crt;
    resolver 8.8.8.8 8.8.4.4 valid=300s;
    resolver_timeout 5s;
}
```

### Generate DH Parameters

```bash
# Generate 4096-bit DH parameters (takes 10-30 minutes)
openssl dhparam -out /etc/ssl/certs/dhparam.pem 4096
```

### Certificate Best Practices

1. **Use Let's Encrypt** for free, automated certificates:
```bash
sudo certbot certonly --nginx -d example.com -d www.example.com
```

2. **Automate renewal**:
```bash
# Add to crontab
0 3 * * * certbot renew --quiet --deploy-hook "systemctl reload nginx"
```

3. **Use ECDSA certificates** (faster than RSA):
```bash
# Generate ECDSA private key
openssl ecparam -genkey -name prime256v1 -out example.com.key
```

4. **Monitor certificate expiry**:
```bash
echo | openssl s_client -connect example.com:443 2>/dev/null | openssl x509 -noout -dates
```

### SSL Testing

**SSL Labs Test**: Aim for A+ rating
```
https://www.ssllabs.com/ssltest/analyze.html?d=example.com
```

**Local testing**:
```bash
# Test TLS 1.3
openssl s_client -connect example.com:443 -tls1_3

# Test OCSP stapling
openssl s_client -connect example.com:443 -status

# Check certificate chain
openssl s_client -connect example.com:443 -showcerts
```

### HTTP/2 Configuration

Enable HTTP/2 for better performance:

```nginx
listen 443 ssl http2;
listen [::]:443 ssl http2;
```

**Note**: HTTP/2 requires TLS. All modern browsers support HTTP/2 over HTTPS.

## Performance Optimization

### 1. Gzip Compression

```nginx
# Enable gzip
gzip on;
gzip_vary on;
gzip_min_length 1024;
gzip_comp_level 6;  # Balance between CPU and compression (1-9)
gzip_types text/plain text/css text/xml text/javascript
           application/json application/javascript application/xml+rss
           application/rss+xml application/atom+xml image/svg+xml;
gzip_disable "msie6";

# Pre-compressed files (if using gzip_static module)
gzip_static on;
```

**Trade-offs**: Higher compression levels (7-9) use more CPU for minimal gains.

### 2. Brotli Compression (Better than Gzip)

If available, enable Brotli (20-25% better compression):

```nginx
brotli on;
brotli_comp_level 6;
brotli_types text/plain text/css text/xml text/javascript
             application/json application/javascript application/xml+rss;
```

**Installation**:
```bash
# Ubuntu/Debian
sudo apt install libnginx-mod-http-brotli-filter libnginx-mod-http-brotli-static
```

### 3. Caching Strategy

#### Browser Caching

```nginx
# Immutable assets (versioned files)
location ~* \.(js|css|png|jpg|jpeg|gif|ico|svg|woff|woff2|ttf)$ {
    expires 1y;
    add_header Cache-Control "public, immutable";
    access_log off;
}

# HTML (don't cache)
location ~* \.html$ {
    add_header Cache-Control "no-cache, no-store, must-revalidate";
    add_header Pragma "no-cache";
    add_header Expires "0";
}
```

#### Proxy Caching

```nginx
# Define cache path
proxy_cache_path /var/cache/nginx levels=1:2 keys_zone=my_cache:10m max_size=1g
                 inactive=60m use_temp_path=off;

server {
    location / {
        proxy_cache my_cache;
        proxy_cache_valid 200 60m;
        proxy_cache_valid 404 10m;
        proxy_cache_use_stale error timeout updating http_500 http_502 http_503 http_504;
        proxy_cache_background_update on;
        proxy_cache_lock on;

        add_header X-Cache-Status $upstream_cache_status;

        proxy_pass http://backend;
    }
}
```

### 4. Connection Optimization

```nginx
# Worker processes (set to number of CPU cores)
worker_processes auto;

# Worker connections (increase for high traffic)
events {
    worker_connections 4096;
    use epoll;  # Linux
    multi_accept on;
}

# Keepalive connections
keepalive_timeout 65;
keepalive_requests 100;

# TCP optimization
tcp_nopush on;
tcp_nodelay on;
```

### 5. File Handle Caching

Dramatically improves performance for static files:

```nginx
open_file_cache max=1000 inactive=20s;
open_file_cache_valid 30s;
open_file_cache_min_uses 2;
open_file_cache_errors on;
```

### 6. Upstream Connection Pooling

For proxied backends:

```nginx
upstream backend {
    server backend1:8080;
    server backend2:8080;

    # Keep connections alive
    keepalive 32;
    keepalive_requests 100;
    keepalive_timeout 60s;
}
```

### 7. Timeouts

Tune based on your application:

```nginx
# Client timeouts
client_body_timeout 12;
client_header_timeout 12;
send_timeout 10;

# Proxy timeouts
proxy_connect_timeout 30s;
proxy_send_timeout 60s;
proxy_read_timeout 60s;
```

**Guideline**: Start conservative, monitor, then adjust.

## Rate Limiting and DDoS Protection

### Basic Rate Limiting

```nginx
# Define rate limit zones
limit_req_zone $binary_remote_addr zone=general:10m rate=10r/s;
limit_req_zone $binary_remote_addr zone=login:10m rate=5r/s;

server {
    # Apply rate limiting
    location / {
        limit_req zone=general burst=20 nodelay;
    }

    location /login {
        limit_req zone=login burst=5 nodelay;
    }
}
```

### Connection Limiting

```nginx
limit_conn_zone $binary_remote_addr zone=addr:10m;

location / {
    limit_conn addr 10;  # Max 10 concurrent connections per IP
}
```

### Advanced: API Rate Limiting by Key

```nginx
# Rate limit by API key
limit_req_zone $http_x_api_key zone=api_key:10m rate=100r/s;

location /api/ {
    # Check API key
    if ($http_x_api_key = "") {
        return 401;
    }

    limit_req zone=api_key burst=50 nodelay;
}
```

### Custom Error Responses

```nginx
# Custom rate limit response
location / {
    limit_req zone=general burst=20 nodelay;
    limit_req_status 429;

    error_page 429 = @ratelimit;
}

location @ratelimit {
    return 429 '{"error":"Rate limit exceeded","retry_after":60}\n';
    add_header Content-Type application/json;
}
```

### Geographic Blocking (GeoIP)

```nginx
# Requires ngx_http_geoip_module
geo $block_country {
    default no;
    CN yes;  # Block China
    RU yes;  # Block Russia
}

server {
    if ($block_country = yes) {
        return 403;
    }
}
```

## Logging and Monitoring

### Enhanced Log Format

```nginx
log_format detailed '$remote_addr - $remote_user [$time_local] '
                   '"$request" $status $body_bytes_sent '
                   '"$http_referer" "$http_user_agent" '
                   'request_time=$request_time '
                   'upstream_response_time=$upstream_response_time '
                   'upstream_addr=$upstream_addr '
                   'upstream_status=$upstream_status '
                   'request_id=$request_id';

access_log /var/log/nginx/access.log detailed;
```

### Conditional Logging

```nginx
# Don't log health checks
map $request_uri $loggable {
    ~^/health 0;
    default 1;
}

access_log /var/log/nginx/access.log detailed if=$loggable;
```

### Error Logging Levels

```nginx
error_log /var/log/nginx/error.log warn;  # Levels: debug, info, notice, warn, error, crit
```

### Log Rotation

Configure logrotate (`/etc/logrotate.d/nginx`):

```
/var/log/nginx/*.log {
    daily
    rotate 14
    missingok
    compress
    delaycompress
    notifempty
    create 0640 www-data adm
    sharedscripts
    postrotate
        [ -f /var/run/nginx.pid ] && kill -USR1 `cat /var/run/nginx.pid`
    endscript
}
```

### Monitoring with Prometheus

Enable Nginx stub status:

```nginx
location /nginx_status {
    stub_status on;
    access_log off;
    allow 127.0.0.1;
    deny all;
}
```

Use `nginx-prometheus-exporter` to expose metrics.

### Real-time Monitoring

```bash
# Monitor access log
tail -f /var/log/nginx/access.log

# Count requests per IP
awk '{print $1}' /var/log/nginx/access.log | sort | uniq -c | sort -rn | head -20

# Monitor response times
awk '{print $NF}' /var/log/nginx/access.log | sort -n | tail -100
```

## High Availability

### Load Balancing

```nginx
upstream backend {
    # Load balancing methods:
    # - round-robin (default)
    # - least_conn (least connections)
    # - ip_hash (session persistence)
    # - hash $variable (custom)

    least_conn;

    server backend1:8080 max_fails=3 fail_timeout=30s weight=1;
    server backend2:8080 max_fails=3 fail_timeout=30s weight=1;
    server backend3:8080 max_fails=3 fail_timeout=30s weight=1;
    server backend4:8080 backup;  # Backup server

    keepalive 32;
}

server {
    location / {
        proxy_pass http://backend;
        proxy_next_upstream error timeout invalid_header http_500 http_502 http_503;
        proxy_next_upstream_tries 3;
    }
}
```

### Health Checks

**Passive** (built-in):
```nginx
server backend1:8080 max_fails=3 fail_timeout=30s;
```

**Active** (Nginx Plus or custom):
```nginx
# Implement health check endpoint in your application
location /health {
    proxy_pass http://backend/health;
    proxy_connect_timeout 5s;
    proxy_read_timeout 5s;
}
```

### Session Persistence

```nginx
upstream backend {
    ip_hash;  # Route same IP to same backend

    # Or use sticky cookie (Nginx Plus)
    # sticky cookie srv_id expires=1h;
}
```

## Common Pitfalls

### 1. Using `if` in location blocks

**Bad**:
```nginx
location / {
    if ($request_uri = "/old-page") {
        rewrite ^ /new-page permanent;
    }
}
```

**Good**:
```nginx
location = /old-page {
    return 301 /new-page;
}
```

**Why**: `if` is evil in location contexts and can cause unexpected behavior.

### 2. Not using `return` for redirects

**Bad**:
```nginx
rewrite ^/(.*)$ https://example.com/$1 permanent;
```

**Good**:
```nginx
return 301 https://example.com$request_uri;
```

**Why**: `return` is faster and more explicit.

### 3. Forgetting `proxy_set_header Host`

**Bad**:
```nginx
proxy_pass http://backend;
```

**Good**:
```nginx
proxy_pass http://backend;
proxy_set_header Host $host;
```

**Why**: Backend needs correct Host header.

### 4. Not handling WebSockets

**Bad**:
```nginx
proxy_pass http://backend;
```

**Good**:
```nginx
proxy_pass http://backend;
proxy_http_version 1.1;
proxy_set_header Upgrade $http_upgrade;
proxy_set_header Connection "upgrade";
```

### 5. Incorrect root vs alias

```nginx
# root appends location to path
location /images/ {
    root /var/www;  # Serves /var/www/images/file.jpg
}

# alias replaces location
location /images/ {
    alias /var/www/pics/;  # Serves /var/www/pics/file.jpg
}
```

### 6. Missing trailing slashes

**Bad**:
```nginx
location /images/ {
    alias /var/www/pics;  # Missing trailing slash!
}
```

**Good**:
```nginx
location /images/ {
    alias /var/www/pics/;  # Trailing slash required
}
```

## Maintenance and Updates

### Keep Nginx Updated

```bash
# Ubuntu/Debian
sudo apt update && sudo apt upgrade nginx

# Check version
nginx -v

# Stable vs mainline
# - Stable: Production use
# - Mainline: Latest features, still stable
```

### Configuration Testing

Always test before reloading:

```bash
# Test configuration
sudo nginx -t

# Reload (zero downtime)
sudo systemctl reload nginx

# Restart (brief downtime)
sudo systemctl restart nginx
```

### Backup Configuration

```bash
# Backup before changes
sudo tar -czf /root/nginx-backup-$(date +%F).tar.gz /etc/nginx/

# Version control
cd /etc/nginx && sudo git init
sudo git add .
sudo git commit -m "Initial nginx config"
```

### Security Updates

Monitor for vulnerabilities:
- Subscribe to Nginx security advisories
- Use `unattended-upgrades` for automatic security patches
- Regular security audits

### Performance Tuning

1. **Baseline measurements**: Use `ab`, `wrk`, or `k6` for benchmarking
2. **Monitor**: CPU, memory, network, disk I/O
3. **Tune incrementally**: Change one thing at a time
4. **Load test**: Always test under realistic load

### Compliance

For regulated industries (PCI-DSS, HIPAA, SOC 2):
- TLS 1.2+ required (TLS 1.3 recommended)
- Strong cipher suites only
- Regular security audits
- Log retention policies
- Access controls
- Certificate management

---

## Quick Reference

### Testing Checklist

- [ ] `nginx -t` passes
- [ ] SSL Labs grade A+
- [ ] Security headers score A+
- [ ] Rate limiting configured
- [ ] Logging enabled
- [ ] Backups in place
- [ ] Monitoring configured
- [ ] Documentation updated

### Security Checklist

- [ ] Server tokens disabled
- [ ] TLS 1.2/1.3 only
- [ ] HSTS enabled
- [ ] Security headers configured
- [ ] Rate limiting enabled
- [ ] Access logs enabled
- [ ] Regular updates scheduled
- [ ] Firewalls configured

### Performance Checklist

- [ ] Gzip/Brotli enabled
- [ ] HTTP/2 enabled
- [ ] Caching configured
- [ ] Keepalive enabled
- [ ] Connection pooling configured
- [ ] File handle caching enabled
- [ ] Worker processes optimized

---

**Last Updated**: January 2025
**Nginx Version**: 1.27.3+
**Standards**: OWASP, Mozilla SSL Config, NIST
