# Kubernetes Manifest Creator - Quick Start Guide

Get started with creating production-ready Kubernetes manifests in minutes.

## Prerequisites

- `kubectl` installed (version 1.28+)
- Access to a Kubernetes cluster (minikube, kind, cloud provider, etc.)
- Basic understanding of Kubernetes concepts

Optional but recommended:
- `kubeconform` for enhanced validation
- `kustomize` for manifest customization
- `helm` for package management

## Installation

### Install kubectl

```bash
# macOS
brew install kubectl

# Linux
curl -LO "https://dl.k8s.io/release/$(curl -L -s https://dl.k8s.io/release/stable.txt)/bin/linux/amd64/kubectl"
chmod +x kubectl
sudo mv kubectl /usr/local/bin/

# Windows (PowerShell)
choco install kubernetes-cli
```

### Install kubeconform (optional)

```bash
# macOS
brew install kubeconform

# Linux
wget https://github.com/yannh/kubeconform/releases/latest/download/kubeconform-linux-amd64.tar.gz
tar xf kubeconform-linux-amd64.tar.gz
sudo mv kubeconform /usr/local/bin/
```

### Clone or Download Templates

```bash
cd /path/to/kubernetes-manifest-creator
chmod +x scripts/*.sh
```

## Quick Examples

### 1. Create a Simple Web Application

**Step 1: Use a template**

```bash
cp templates/deployment/web-app-deployment.yaml my-app-deployment.yaml
```

**Step 2: Customize for your app**

Edit `my-app-deployment.yaml`:
- Change `name: web-app` to `name: my-app`
- Update `image: nginx:1.25-alpine` to your image
- Adjust replicas, resources, ports as needed
- Update labels to match your app

**Step 3: Create a service**

```bash
cp templates/service/clusterip-service.yaml my-app-service.yaml
```

Edit to match your deployment labels and ports.

**Step 4: Validate**

```bash
./scripts/validate-k8s.sh my-app-deployment.yaml my-app-service.yaml
```

**Step 5: Apply**

```bash
kubectl apply -f my-app-deployment.yaml
kubectl apply -f my-app-service.yaml
```

### 2. Interactive Generator

Use the interactive script to create manifests with prompts:

```bash
./scripts/generate-manifests.sh
```

Follow the prompts to configure:
- Resource type (Deployment, Service, etc.)
- Application name
- Container image
- Replicas
- Resource limits
- Ports
- Environment variables

### 3. Complete Application Stack

Deploy a full 3-tier application:

```bash
kubectl apply -f templates/full-app-example/complete-3tier-app.yaml
```

This creates:
- Namespace with Pod Security Standards
- PostgreSQL database (StatefulSet)
- Node.js API backend (Deployment)
- React frontend (Deployment)
- Services for all components
- Ingress with TLS
- ConfigMaps and Secrets
- Pod Disruption Budgets
- Horizontal Pod Autoscaler
- Network Policies

**Customize before deploying:**

```bash
cp templates/full-app-example/complete-3tier-app.yaml my-stack.yaml
# Edit my-stack.yaml with your configuration
./scripts/validate-k8s.sh my-stack.yaml
kubectl apply -f my-stack.yaml
```

## Common Workflows

### Deploying a Microservice

1. **Create deployment**
   ```bash
   ./scripts/generate-manifests.sh
   # Select: 1. Deployment
   # Answer prompts
   ```

2. **Create service**
   ```bash
   cp templates/service/clusterip-service.yaml my-service.yaml
   # Edit to match deployment
   ```

3. **Add configuration**
   ```bash
   cp templates/configmap/app-config.yaml my-config.yaml
   # Edit with your config
   ```

4. **Validate and apply**
   ```bash
   ./scripts/validate-k8s.sh my-*.yaml
   kubectl apply -f my-deployment.yaml
   kubectl apply -f my-service.yaml
   kubectl apply -f my-config.yaml
   ```

### Adding Ingress for HTTPS

1. **Copy ingress template**
   ```bash
   cp templates/ingress/nginx-ingress-tls.yaml my-ingress.yaml
   ```

2. **Edit configuration**
   - Update `host: example.com` to your domain
   - Update service name and port
   - Configure TLS secret name
   - Add annotations as needed

3. **Install cert-manager (for TLS)**
   ```bash
   kubectl apply -f https://github.com/cert-manager/cert-manager/releases/download/v1.13.0/cert-manager.yaml
   ```

4. **Apply ingress**
   ```bash
   kubectl apply -f my-ingress.yaml
   ```

### Deploying a Stateful Application

For databases, caches, or other stateful apps:

1. **Use StatefulSet template**
   ```bash
   # For PostgreSQL
   cp templates/statefulset/postgresql-statefulset.yaml my-db.yaml

   # For Redis
   cp templates/statefulset/redis-statefulset.yaml my-cache.yaml
   ```

2. **Customize**
   - Update storage size
   - Set resource limits
   - Configure passwords in secrets
   - Adjust replicas

3. **Apply**
   ```bash
   kubectl apply -f my-db.yaml
   ```

## Validation

### Validate Before Apply

Always validate manifests before deploying:

```bash
# Single file
./scripts/validate-k8s.sh deployment.yaml

# Multiple files
./scripts/validate-k8s.sh *.yaml

# Directory
./scripts/validate-k8s.sh manifests/

# Specific Kubernetes version
KUBE_VERSION=1.29.0 ./scripts/validate-k8s.sh manifests/
```

### Dry Run

Test apply without making changes:

```bash
kubectl apply -f deployment.yaml --dry-run=client
kubectl apply -f deployment.yaml --dry-run=server
```

### Diff

See what would change:

```bash
kubectl diff -f deployment.yaml
```

## Customization with Kustomize

Create environment-specific configurations:

### Directory structure

```
manifests/
├── base/
│   ├── deployment.yaml
│   ├── service.yaml
│   └── kustomization.yaml
├── overlays/
│   ├── development/
│   │   └── kustomization.yaml
│   └── production/
│       └── kustomization.yaml
```

### base/kustomization.yaml

```yaml
apiVersion: kustomize.config.k8s.io/v1beta1
kind: Kustomization

resources:
- deployment.yaml
- service.yaml

commonLabels:
  app: myapp
```

### overlays/production/kustomization.yaml

```yaml
apiVersion: kustomize.config.k8s.io/v1beta1
kind: Kustomization

bases:
- ../../base

namespace: production

commonLabels:
  env: production

replicas:
- name: myapp
  count: 5

images:
- name: myapp
  newTag: v1.2.3
```

### Apply with kustomize

```bash
# Development
kubectl apply -k overlays/development

# Production
kubectl apply -k overlays/production
```

## Debugging

### Check pod status

```bash
kubectl get pods
kubectl describe pod <pod-name>
kubectl logs <pod-name>
kubectl logs <pod-name> --previous  # Previous container logs
```

### Check events

```bash
kubectl get events --sort-by='.lastTimestamp'
```

### Exec into container

```bash
kubectl exec -it <pod-name> -- /bin/sh
```

### Port forwarding

```bash
kubectl port-forward <pod-name> 8080:80
```

## Common Issues and Solutions

### Pod stays in Pending

**Cause**: Insufficient resources or scheduling issues

**Solution**:
```bash
kubectl describe pod <pod-name>
# Check events for scheduling errors
# Verify resource requests are not too high
# Check node capacity: kubectl describe nodes
```

### CrashLoopBackOff

**Cause**: Application crashes on startup

**Solution**:
```bash
kubectl logs <pod-name>
# Check startup probes
# Verify environment variables and secrets
# Check file permissions
```

### ImagePullBackOff

**Cause**: Cannot pull container image

**Solution**:
```bash
kubectl describe pod <pod-name>
# Verify image name and tag
# Check imagePullSecrets
# Verify registry access
```

### Permission Denied

**Cause**: Security context issues

**Solution**:
- Verify `runAsUser` and `fsGroup`
- Check volume permissions
- Ensure user exists in container
- Use `fsGroup` for volume access

## Best Practices Summary

### Before Deploying

- [ ] Validate manifests with `./scripts/validate-k8s.sh`
- [ ] Review security settings (non-root, read-only FS)
- [ ] Set resource requests and limits
- [ ] Configure health probes
- [ ] Test in development first
- [ ] Review labels and annotations

### Security Checklist

- [ ] `runAsNonRoot: true`
- [ ] `readOnlyRootFilesystem: true`
- [ ] `allowPrivilegeEscalation: false`
- [ ] `capabilities.drop: [ALL]`
- [ ] Secrets in Kubernetes Secrets (not manifests)
- [ ] NetworkPolicies defined

### Reliability Checklist

- [ ] At least 2-3 replicas
- [ ] Resource limits prevent resource exhaustion
- [ ] Health probes configured
- [ ] PodDisruptionBudget for critical apps
- [ ] Graceful shutdown implemented

## Next Steps

1. **Read the full best practices guide**
   ```bash
   cat resources/best-practices.md
   ```

2. **Review security checklist**
   ```bash
   cat resources/security-checklist.md
   ```

3. **Explore templates**
   ```bash
   ls -R templates/
   ```

4. **Set up monitoring**
   - Install Prometheus and Grafana
   - Configure metrics endpoints
   - Set up alerting

5. **Implement GitOps**
   - Store manifests in Git
   - Use ArgoCD or Flux
   - Automate deployments

## Getting Help

- **Kubernetes Documentation**: https://kubernetes.io/docs/
- **kubectl Cheat Sheet**: https://kubernetes.io/docs/reference/kubectl/cheatsheet/
- **Community**: https://kubernetes.io/community/

## Example: End-to-End Deployment

Here's a complete example deploying a web app:

```bash
# 1. Create namespace
kubectl create namespace myapp

# 2. Create ConfigMap
cat <<EOF | kubectl apply -f -
apiVersion: v1
kind: ConfigMap
metadata:
  name: myapp-config
  namespace: myapp
data:
  APP_ENV: production
  LOG_LEVEL: info
EOF

# 3. Create Secret
kubectl create secret generic myapp-secrets \
  --from-literal=api-key=super-secret \
  --namespace=myapp

# 4. Copy and customize deployment
cp templates/deployment/web-app-deployment.yaml myapp-deployment.yaml
# Edit myapp-deployment.yaml with your settings

# 5. Copy and customize service
cp templates/service/clusterip-service.yaml myapp-service.yaml
# Edit myapp-service.yaml

# 6. Validate
./scripts/validate-k8s.sh myapp-deployment.yaml myapp-service.yaml

# 7. Apply
kubectl apply -f myapp-deployment.yaml
kubectl apply -f myapp-service.yaml

# 8. Check status
kubectl get pods -n myapp
kubectl get svc -n myapp

# 9. View logs
kubectl logs -n myapp -l app.kubernetes.io/name=myapp

# 10. Access the app
kubectl port-forward -n myapp svc/myapp 8080:80
# Open http://localhost:8080 in browser
```

Congratulations! You've deployed your first Kubernetes application with production-ready configuration.
