# Helm Chart Generator - Quick Start

Get started with Helm chart generation in under 5 minutes.

## Installation

No installation required! The scripts work with standard Helm 3.x.

### Prerequisites
- Helm 3.x installed ([Install Guide](https://helm.sh/docs/intro/install/))
- Kubernetes cluster (optional for testing)

## Quick Start

### 1. Generate a Chart

**Interactive Mode:**
```bash
cd scripts/
./generate-helm-chart.sh
```

**Command Line Mode:**
```bash
./generate-helm-chart.sh -t web-application -n my-app
```

### 2. Customize Values

Edit the generated `values.yaml`:
```bash
cd my-app/
vim values.yaml
```

Key configurations to customize:
- `image.repository` - Your container image
- `image.tag` - Image version
- `replicaCount` - Number of pods
- `service.port` - Service port
- `ingress.enabled` - Enable external access

### 3. Validate Chart

```bash
../validate-helm-chart.sh .
```

Or use Helm commands directly:
```bash
helm lint .
helm template . --debug
```

### 4. Install Chart

**Dry run first:**
```bash
helm install my-app . --dry-run --debug
```

**Install for real:**
```bash
helm install my-app .
```

### 5. Access Application

Get access instructions from NOTES:
```bash
helm get notes my-app
```

## Chart Types Overview

### web-application
**Best for:** Web applications, REST APIs, frontend apps

**Includes:**
- Deployment with configurable replicas
- Service (ClusterIP/NodePort/LoadBalancer)
- Optional Ingress
- Optional Horizontal Pod Autoscaler
- Health probes
- Security contexts

**Quick example:**
```bash
./generate-helm-chart.sh -t web-application -n frontend-app
cd frontend-app/
# Edit values.yaml - set your image
helm install frontend ./
```

### stateful-application
**Best for:** Databases, caches, stateful services

**Includes:**
- StatefulSet with stable identity
- Persistent Volume Claims
- Headless service
- Ordered pod management

**Quick example:**
```bash
./generate-helm-chart.sh -t stateful-application -n postgres
cd postgres/
# Edit values.yaml - configure persistence
helm install postgres ./
```

### microservice
**Best for:** Lightweight backend services

**Includes:**
- Minimal deployment configuration
- Service discovery
- Auto-scaling enabled by default
- Optimized resource limits

**Quick example:**
```bash
./generate-helm-chart.sh -t microservice -n auth-service
cd auth-service/
helm install auth-service ./
```

### cronjob
**Best for:** Scheduled tasks, batch jobs, backups

**Includes:**
- CronJob resource
- Configurable schedule
- Job history limits
- TTL for finished jobs

**Quick example:**
```bash
./generate-helm-chart.sh -t cronjob -n backup-job
cd backup-job/
# Edit values.yaml - set schedule: "0 2 * * *"  # Daily at 2 AM
helm install backup-job ./
```

### redis
**Best for:** Redis cache deployments

**Includes:**
- Redis deployment
- Persistent storage
- Redis configuration
- Service for connections

**Quick example:**
```bash
./generate-helm-chart.sh -t redis -n app-cache
cd app-cache/
helm install app-cache ./
```

## Common Workflows

### Development

```bash
# Create chart
./generate-helm-chart.sh -t web-application -n dev-app

# Edit and test
cd dev-app/
helm template . | kubectl apply --dry-run=client -f -

# Install
helm install dev-app .

# Make changes and upgrade
helm upgrade dev-app .

# Check status
helm status dev-app
kubectl get all -l app.kubernetes.io/instance=dev-app
```

### Production Deployment

```bash
# Create chart with production values
./generate-helm-chart.sh -t web-application -n prod-app
cd prod-app/

# Create production values
cat > values-prod.yaml <<EOF
replicaCount: 5

image:
  repository: myregistry/myapp
  tag: "v1.2.3"

resources:
  limits:
    cpu: 1000m
    memory: 1Gi
  requests:
    cpu: 500m
    memory: 512Mi

ingress:
  enabled: true
  className: nginx
  hosts:
    - host: app.example.com
      paths:
        - path: /
          pathType: Prefix
  tls:
    - secretName: app-tls
      hosts:
        - app.example.com

autoscaling:
  enabled: true
  minReplicas: 5
  maxReplicas: 20
  targetCPUUtilizationPercentage: 70
EOF

# Validate
../validate-helm-chart.sh .

# Install to production namespace
helm install prod-app . \
  --namespace production \
  --create-namespace \
  --values values-prod.yaml
```

### Multi-Environment Setup

```bash
# Create chart once
./generate-helm-chart.sh -t web-application -n myapp
cd myapp/

# Create environment-specific values
cat > values-dev.yaml <<EOF
replicaCount: 1
ingress:
  hosts:
    - host: dev.myapp.local
EOF

cat > values-staging.yaml <<EOF
replicaCount: 2
ingress:
  hosts:
    - host: staging.myapp.com
EOF

cat > values-prod.yaml <<EOF
replicaCount: 5
autoscaling:
  enabled: true
ingress:
  hosts:
    - host: myapp.com
EOF

# Deploy to environments
helm install myapp . -f values-dev.yaml -n dev
helm install myapp . -f values-staging.yaml -n staging
helm install myapp . -f values-prod.yaml -n production
```

## Useful Commands

### Chart Management
```bash
# List installed charts
helm list

# Get chart status
helm status <release-name>

# Get chart values
helm get values <release-name>

# Upgrade chart
helm upgrade <release-name> .

# Rollback
helm rollback <release-name> <revision>

# Uninstall
helm uninstall <release-name>
```

### Debugging
```bash
# Lint chart
helm lint .

# Render templates
helm template .

# Debug specific values
helm template . --set image.tag=v2.0

# Show all computed values
helm template . --debug

# Test installation without deploying
helm install --dry-run --debug myapp .
```

### Testing
```bash
# Run Helm tests
helm test <release-name>

# Get logs from test pods
kubectl logs -l helm.sh/hook=test
```

## Customization Tips

### Override Values
```bash
# From command line
helm install myapp . --set image.tag=v2.0,replicaCount=3

# From file
helm install myapp . -f custom-values.yaml

# Multiple files (later files override earlier)
helm install myapp . -f base.yaml -f override.yaml
```

### Template Values File
```yaml
# custom-values.yaml
image:
  repository: myregistry/myapp
  tag: "{{ .Values.version }}"

ingress:
  enabled: true
  hosts:
    - host: "{{ .Values.environment }}.myapp.com"
```

## Next Steps

1. **Read Best Practices** - See `resources/best-practices.md`
2. **Customize Templates** - Edit files in `templates/` directory
3. **Add Dependencies** - Define in `Chart.yaml`
4. **Package Chart** - Run `helm package .`
5. **Share Chart** - Push to chart repository

## Troubleshooting

### Chart Won't Install
```bash
# Check validation
helm lint .

# Verify templates render
helm template . --debug

# Check Kubernetes API versions
kubectl api-versions
```

### Pods Not Starting
```bash
# Check pod status
kubectl get pods -l app.kubernetes.io/instance=<release>

# Check pod logs
kubectl logs <pod-name>

# Describe pod
kubectl describe pod <pod-name>
```

### Values Not Applied
```bash
# Verify computed values
helm get values <release-name> --all

# Check template rendering
helm template <release-name> . --debug
```

## Resources

- [Helm Documentation](https://helm.sh/docs/)
- [Helm Best Practices](https://helm.sh/docs/chart_best_practices/)
- [Kubernetes Documentation](https://kubernetes.io/docs/)
- Chart Templates: `templates/` directory
- Best Practices: `resources/best-practices.md`

## Getting Help

```bash
# Generator help
./generate-helm-chart.sh --help

# Validator help
./validate-helm-chart.sh --help

# Helm help
helm --help
helm install --help
```
