# Helm Chart Best Practices (2025)

Comprehensive guide to creating production-ready Helm charts following current standards and best practices.

## Chart Structure

### Standard Layout
```
mychart/
├── Chart.yaml              # Chart metadata
├── values.yaml             # Default configuration
├── values.schema.json      # Values validation (optional but recommended)
├── templates/
│   ├── NOTES.txt          # Post-install instructions
│   ├── _helpers.tpl       # Template helpers
│   ├── deployment.yaml    # Main workload
│   ├── service.yaml       # Service definition
│   ├── ingress.yaml       # Ingress (if applicable)
│   ├── hpa.yaml           # Autoscaling (if applicable)
│   ├── serviceaccount.yaml
│   └── tests/
│       └── test-connection.yaml
├── charts/                 # Subcharts directory
└── README.md              # User documentation
```

## Chart.yaml Best Practices

### Helm 3 Format (API v2)
```yaml
apiVersion: v2
name: my-application
description: Clear, concise description of what this chart does
type: application  # or 'library'
version: 1.2.3     # Chart version (SemVer)
appVersion: "2.0.1"  # Application version (quoted)

# Recommended fields
keywords:
  - web
  - backend
  - microservice

maintainers:
  - name: Team Name
    email: team@example.com
    url: https://example.com

# Dependencies
dependencies:
  - name: postgresql
    version: "12.x.x"
    repository: https://charts.bitnami.com/bitnami
    condition: postgresql.enabled

# Kubernetes version constraint
kubeVersion: ">=1.24.0-0"

# Annotations for additional metadata
annotations:
  category: Infrastructure
```

### Version Management
- **Chart version**: Increment when chart changes
  - MAJOR: Breaking changes
  - MINOR: New features, backward compatible
  - PATCH: Bug fixes
- **App version**: Track application version independently

## values.yaml Best Practices

### 1. Provide Sensible Defaults
```yaml
# ✓ Good - Works out of the box
replicaCount: 2
image:
  repository: nginx
  tag: "1.25.0"
  pullPolicy: IfNotPresent

# ✗ Bad - Requires configuration
replicaCount: null  # User must provide
```

### 2. Use Hierarchical Structure
```yaml
# ✓ Good - Organized by component
image:
  repository: myapp
  tag: "1.0.0"
  pullPolicy: IfNotPresent

service:
  type: ClusterIP
  port: 80
  annotations: {}

# ✗ Bad - Flat structure
imageRepository: myapp
imageTag: "1.0.0"
serviceType: ClusterIP
servicePort: 80
```

### 3. Document All Values
```yaml
# Number of replicas for the deployment
# For production, use at least 2 for high availability
replicaCount: 2

# Container image configuration
image:
  # Docker image repository
  repository: nginx
  # Image pull policy: Always, IfNotPresent, Never
  pullPolicy: IfNotPresent
  # Overrides the image tag (default is Chart.appVersion)
  tag: ""
```

### 4. Support Common Customizations
```yaml
# Enable optional features with flags
ingress:
  enabled: false  # Set to true to create ingress

autoscaling:
  enabled: false  # Set to true to enable HPA

# Allow custom labels and annotations
podLabels: {}
podAnnotations: {}
```

## Template Best Practices

### 1. Use Named Templates (_helpers.tpl)
```yaml
{{/*
Expand the name of the chart.
*/}}
{{- define "mychart.name" -}}
{{- default .Chart.Name .Values.nameOverride | trunc 63 | trimSuffix "-" }}
{{- end }}

{{/*
Common labels
*/}}
{{- define "mychart.labels" -}}
helm.sh/chart: {{ include "mychart.chart" . }}
{{ include "mychart.selectorLabels" . }}
{{- if .Chart.AppVersion }}
app.kubernetes.io/version: {{ .Chart.AppVersion | quote }}
{{- end }}
app.kubernetes.io/managed-by: {{ .Release.Service }}
{{- end }}
```

### 2. Conditional Resource Creation
```yaml
{{- if .Values.ingress.enabled -}}
apiVersion: networking.k8s.io/v1
kind: Ingress
metadata:
  name: {{ include "mychart.fullname" . }}
...
{{- end }}
```

### 3. Proper Indentation with nindent/indent
```yaml
# ✓ Good - Proper indentation
metadata:
  labels:
    {{- include "mychart.labels" . | nindent 4 }}

# ✗ Bad - Manual spacing prone to errors
metadata:
  labels:
{{ include "mychart.labels" . }}
```

### 4. Validate Required Values
```yaml
{{- if not .Values.database.password }}
{{- fail "database.password is required" }}
{{- end }}

# Or use 'required' function
env:
  - name: DB_PASSWORD
    value: {{ required "database.password is required" .Values.database.password }}
```

## Security Best Practices

### 1. Run as Non-Root User
```yaml
podSecurityContext:
  runAsNonRoot: true
  runAsUser: 1000
  fsGroup: 1000
  seccompProfile:
    type: RuntimeDefault

securityContext:
  allowPrivilegeEscalation: false
  capabilities:
    drop:
    - ALL
  readOnlyRootFilesystem: true
  runAsNonRoot: true
  runAsUser: 1000
```

### 2. Resource Limits
```yaml
# ✓ Always specify limits and requests
resources:
  limits:
    cpu: 200m
    memory: 256Mi
  requests:
    cpu: 100m
    memory: 128Mi

# ✗ Don't leave unlimited
resources: {}
```

### 3. Network Policies
```yaml
# Include network policies for production
{{- if .Values.networkPolicy.enabled }}
apiVersion: networking.k8s.io/v1
kind: NetworkPolicy
metadata:
  name: {{ include "mychart.fullname" . }}
spec:
  podSelector:
    matchLabels:
      {{- include "mychart.selectorLabels" . | nindent 6 }}
  policyTypes:
    - Ingress
    - Egress
  ingress:
    - from:
        - podSelector: {}
  egress:
    - to:
        - podSelector: {}
{{- end }}
```

### 4. Service Account
```yaml
# Create dedicated service account
serviceAccount:
  create: true
  annotations: {}
  # Don't auto-mount token unless needed
  automountServiceAccountToken: false
```

## Reliability Best Practices

### 1. Health Probes
```yaml
# Always include both probes
livenessProbe:
  httpGet:
    path: /healthz
    port: http
  initialDelaySeconds: 30
  periodSeconds: 10
  timeoutSeconds: 5
  failureThreshold: 3

readinessProbe:
  httpGet:
    path: /ready
    port: http
  initialDelaySeconds: 10
  periodSeconds: 5
  timeoutSeconds: 3
  failureThreshold: 3
```

### 2. Pod Disruption Budgets
```yaml
{{- if .Values.podDisruptionBudget.enabled }}
apiVersion: policy/v1
kind: PodDisruptionBudget
metadata:
  name: {{ include "mychart.fullname" . }}
spec:
  minAvailable: {{ .Values.podDisruptionBudget.minAvailable }}
  selector:
    matchLabels:
      {{- include "mychart.selectorLabels" . | nindent 6 }}
{{- end }}
```

### 3. Topology Spread Constraints
```yaml
# Spread pods across zones for HA
topologySpreadConstraints:
  - maxSkew: 1
    topologyKey: topology.kubernetes.io/zone
    whenUnsatisfiable: DoNotSchedule
    labelSelector:
      matchLabels:
        {{- include "mychart.selectorLabels" . | nindent 8 }}
```

## Common Mistakes to Avoid

### 1. Over-Templatization
```yaml
# ✗ Bad - Too flexible, hard to maintain
{{- if .Values.deployment.strategy.type }}
  {{- if eq .Values.deployment.strategy.type "RollingUpdate" }}
    {{- if .Values.deployment.strategy.rollingUpdate }}
      {{- if .Values.deployment.strategy.rollingUpdate.maxSurge }}
...

# ✓ Good - Sensible defaults, simple override
strategy:
  {{- toYaml .Values.deployment.strategy | nindent 4 }}
```

### 2. Hardcoded Values
```yaml
# ✗ Bad - Hardcoded namespace
namespace: production

# ✓ Good - Use Release namespace
namespace: {{ .Release.Namespace }}
```

### 3. Missing Required Fields
```yaml
# ✗ Bad - Missing selector
apiVersion: apps/v1
kind: Deployment
metadata:
  name: myapp

# ✓ Good - Complete spec
spec:
  replicas: {{ .Values.replicaCount }}
  selector:
    matchLabels:
      {{- include "mychart.selectorLabels" . | nindent 6 }}
```

## Testing

### 1. Template Tests
```yaml
# templates/tests/test-connection.yaml
apiVersion: v1
kind: Pod
metadata:
  name: "{{ include "mychart.fullname" . }}-test"
  annotations:
    "helm.sh/hook": test
spec:
  containers:
    - name: wget
      image: busybox
      command: ['wget']
      args: ['{{ include "mychart.fullname" . }}:{{ .Values.service.port }}']
  restartPolicy: Never
```

### 2. Validation Commands
```bash
# Lint chart
helm lint .

# Render templates
helm template . --debug

# Test with custom values
helm template . -f test-values.yaml

# Dry run install
helm install --dry-run --debug myrelease .

# Run tests
helm test myrelease
```

## Documentation

### 1. README.md Structure
```markdown
# Chart Name

## Overview
Brief description of what this chart deploys

## Prerequisites
- Kubernetes 1.24+
- Helm 3.x

## Installation
\`\`\`bash
helm install my-release .
\`\`\`

## Configuration
| Parameter | Description | Default |
|-----------|-------------|---------|
| `replicaCount` | Number of replicas | `2` |

## Upgrading
Notes on upgrades and breaking changes

## Uninstalling
\`\`\`bash
helm uninstall my-release
\`\`\`
```

### 2. NOTES.txt
```
1. Get the application URL:
{{- if .Values.ingress.enabled }}
  http{{ if .Values.ingress.tls }}s{{ end }}://{{ index .Values.ingress.hosts 0 "host" }}
{{- else }}
  kubectl port-forward svc/{{ include "mychart.fullname" . }} 8080:80
{{- end }}

2. Check pod status:
  kubectl get pods -l app.kubernetes.io/instance={{ .Release.Name }}
```

## Version Compatibility

### Helm 3 vs Helm 2
- Use `apiVersion: v2` in Chart.yaml
- No more Tiller - direct Kubernetes API
- Three-way strategic merge patches
- Secrets as default storage
- Automatic CRD management
- JSONSchema values validation

### Kubernetes API Versions (2025)
- Ingress: `networking.k8s.io/v1` (not beta)
- PodDisruptionBudget: `policy/v1` (not v1beta1)
- HorizontalPodAutoscaler: `autoscaling/v2` (not v2beta2)
- CronJob: `batch/v1` (not v1beta1)

## References
- [Official Helm Best Practices](https://helm.sh/docs/chart_best_practices/)
- [Helm Chart Template Guide](https://helm.sh/docs/chart_template_guide/)
- [Kubernetes API Reference](https://kubernetes.io/docs/reference/)
