# Docker Compose Best Practices (2025)

Comprehensive guide for creating production-ready Docker Compose configurations.

## Modern Compose v2 Format

### No Version Field
```yaml
# ✓ Good - Compose v2 (no version field)
services:
  app:
    image: nginx:1.25

# ✗ Bad - Outdated (Compose v1 style)
version: '3.8'
services:
  app:
    image: nginx
```

**Why**: Docker Compose v2 ignores the version field. Using it triggers warnings.

### Use compose.yaml
- Modern: `compose.yaml`
- Legacy: `docker-compose.yml`

Use `docker compose` (no hyphen) instead of `docker-compose`.

## Image Management

### Always Use Explicit Tags
```yaml
# ✓ Good
image: postgres:16.1-alpine
image: redis:7.2-alpine
image: nginx:1.25.3

# ✗ Bad
image: postgres:latest
image: redis
```

**Impact**: :latest tags are unpredictable and can break deployments.

### Multi-Stage Builds
```dockerfile
# Dockerfile
FROM node:20 AS builder
WORKDIR /app
COPY package*.json ./
RUN npm ci --production
COPY . .
RUN npm run build

FROM node:20-alpine
WORKDIR /app
COPY --from=builder /app/dist ./dist
COPY --from=builder /app/node_modules ./node_modules
CMD ["node", "dist/server.js"]
```

## Service Configuration

### Resource Limits (Critical!)
```yaml
services:
  app:
    deploy:
      resources:
        limits:
          cpus: '1.0'
          memory: 512M
        reservations:
          cpus: '0.5'
          memory: 256M
```

**Why**: Without limits, one container can consume all system resources.

### Health Checks
```yaml
healthcheck:
  test: ["CMD", "curl", "-f", "http://localhost/health"]
  interval: 30s
  timeout: 10s
  retries: 3
  start_period: 40s
```

**Database Example:**
```yaml
healthcheck:
  test: ["CMD-SHELL", "pg_isready -U postgres"]
  interval: 10s
  timeout: 5s
  retries: 5
```

### Restart Policies
```yaml
# Production
restart: unless-stopped

# Development
restart: "no"

# Always (use cautiously)
restart: always
```

### Service Dependencies
```yaml
# ✓ Good - Wait for healthy state
depends_on:
  db:
    condition: service_healthy
  redis:
    condition: service_started

# ✗ Bad - Just start order (doesn't wait for ready)
depends_on:
  - db
  - redis
```

## Security Best Practices

### 1. No Hardcoded Secrets
```yaml
# ✓ Good - Use env files
env_file:
  - .env
environment:
  - DB_HOST=postgres
  - DB_NAME=${POSTGRES_DB}

# ✗ Bad - Hardcoded
environment:
  - DB_PASSWORD=supersecret123
```

### 2. Run as Non-Root
```yaml
user: "1000:1000"  # UID:GID

# Or in Dockerfile
USER appuser
```

### 3. Read-Only Root Filesystem
```yaml
security_opt:
  - no-new-privileges:true
read_only: true
tmpfs:
  - /tmp
  - /var/run
```

### 4. Drop Capabilities
```yaml
cap_drop:
  - ALL
cap_add:
  - NET_BIND_SERVICE  # Only if needed
```

### 5. Minimal Permissions on .env
```bash
chmod 600 .env
# Add to .gitignore
echo ".env" >> .gitignore
```

## Networking

### Network Isolation
```yaml
networks:
  frontend:
    driver: bridge
  backend:
    driver: bridge
    internal: true  # No external access

services:
  web:
    networks:
      - frontend

  api:
    networks:
      - frontend
      - backend

  db:
    networks:
      - backend  # Isolated from frontend
```

### Named Networks
```yaml
networks:
  app-network:
    driver: bridge
    ipam:
      config:
        - subnet: 172.20.0.0/24
```

## Volume Management

### Named Volumes (Recommended)
```yaml
volumes:
  postgres-data:
    driver: local
  redis-data:
    driver: local

services:
  db:
    volumes:
      - postgres-data:/var/lib/postgresql/data
```

### Bind Mounts (Development)
```yaml
# Development only
volumes:
  - ./app:/app          # Code for hot-reload
  - /app/node_modules   # Preserve node_modules
```

### Volume Backups
```bash
# Backup
docker run --rm -v postgres-data:/data -v $(pwd):/backup \
  alpine tar czf /backup/backup.tar.gz /data

# Restore
docker run --rm -v postgres-data:/data -v $(pwd):/backup \
  alpine tar xzf /backup/backup.tar.gz -C /
```

## Environment Variables

### .env File Structure
```bash
# Database
POSTGRES_USER=myuser
POSTGRES_PASSWORD=change-this-password
POSTGRES_DB=mydb

# Application
NODE_ENV=production
APP_SECRET=generate-random-string
API_PORT=3000

# External Services
REDIS_URL=redis://redis:6379
SMTP_HOST=smtp.example.com
```

### Variable Interpolation
```yaml
environment:
  - DATABASE_URL=postgresql://${POSTGRES_USER}:${POSTGRES_PASSWORD}@db:5432/${POSTGRES_DB}
```

## Logging

### Configure Logging Driver
```yaml
logging:
  driver: "json-file"
  options:
    max-size: "10m"
    max-file: "3"
```

### Centralized Logging
```yaml
logging:
  driver: "syslog"
  options:
    syslog-address: "tcp://192.168.0.42:123"
```

## Development vs Production

### Development Configuration
```yaml
# compose.override.yaml (auto-loaded in dev)
services:
  app:
    build:
      context: .
      target: development
    volumes:
      - ./app:/app
    environment:
      - NODE_ENV=development
      - DEBUG=*
    command: npm run dev
```

### Production Configuration
```yaml
# compose.prod.yaml
services:
  app:
    image: myapp:1.0.0
    restart: unless-stopped
    deploy:
      replicas: 3
      resources:
        limits:
          cpus: '1.0'
          memory: 1G
```

Usage:
```bash
# Development
docker compose up

# Production
docker compose -f compose.yaml -f compose.prod.yaml up -d
```

## Profiles for Optional Services

```yaml
services:
  app:
    image: myapp:latest

  # Only start with --profile debug
  debug-tools:
    image: nicolaka/netshoot
    profiles:
      - debug

  # Only start with --profile admin
  phpmyadmin:
    image: phpmyadmin:latest
    profiles:
      - admin
```

Usage:
```bash
docker compose up              # Only app
docker compose --profile debug up   # app + debug-tools
docker compose --profile admin up   # app + phpmyadmin
```

## Common Patterns

### Database with Initialization
```yaml
db:
  image: postgres:16-alpine
  volumes:
    - postgres-data:/var/lib/postgresql/data
    - ./init.sql:/docker-entrypoint-initdb.d/init.sql:ro
```

### Service with Build Args
```yaml
app:
  build:
    context: .
    args:
      - BUILD_DATE=$(date -u +'%Y-%m-%dT%H:%M:%SZ')
      - VCS_REF=$(git rev-parse --short HEAD)
```

### Multiple Commands
```yaml
command: >
  sh -c "
  python manage.py migrate &&
  python manage.py collectstatic --noinput &&
  gunicorn myapp.wsgi:application --bind 0.0.0.0:8000
  "
```

## Performance Optimization

### Build Cache
```bash
# Use BuildKit for better caching
DOCKER_BUILDKIT=1 docker compose build
```

### Layer Caching
```dockerfile
# Copy dependencies first (cached)
COPY package*.json ./
RUN npm ci

# Copy code last (changes frequently)
COPY . .
```

### Multi-Platform Builds
```yaml
build:
  context: .
  platforms:
    - linux/amd64
    - linux/arm64
```

## Troubleshooting

### Debug Mode
```bash
docker compose up --verbose
docker compose config
docker compose logs -f service-name
```

### Container Shell Access
```bash
docker compose exec service-name sh
docker compose exec -u root service-name bash
```

### Network Debugging
```bash
docker compose exec app ping db
docker compose exec app nslookup db
docker network inspect project_network
```

### Volume Issues
```bash
docker volume ls
docker volume inspect volume-name
docker compose down -v  # Remove volumes
```

## CI/CD Integration

### GitHub Actions
```yaml
name: Deploy
on:
  push:
    branches: [main]
jobs:
  deploy:
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v4
      - name: Deploy
        run: |
          docker compose -f compose.yaml -f compose.prod.yaml up -d
```

### GitLab CI
```yaml
deploy:
  stage: deploy
  script:
    - docker compose -f compose.yaml -f compose.prod.yaml up -d
  only:
    - main
```

## Monitoring

### Health Check Endpoints
```javascript
// Express.js example
app.get('/health', (req, res) => {
  res.status(200).json({ status: 'healthy' });
});
```

### Metrics Collection
```yaml
prometheus:
  image: prom/prometheus
  volumes:
    - ./prometheus.yml:/etc/prometheus/prometheus.yml
  ports:
    - "9090:9090"
```

## Checklist for Production

- [ ] No :latest tags
- [ ] All passwords in .env file
- [ ] .env file chmod 600
- [ ] .env in .gitignore
- [ ] Resource limits defined
- [ ] Health checks configured
- [ ] Restart policies set
- [ ] Named volumes used
- [ ] Networks for isolation
- [ ] depends_on with conditions
- [ ] No hardcoded secrets
- [ ] Run as non-root user
- [ ] Logging configured
- [ ] Backups automated
- [ ] Monitoring enabled

## References

- [Official Docker Compose Docs](https://docs.docker.com/compose/)
- [Compose File Specification](https://docs.docker.com/compose/compose-file/)
- [Docker Security Best Practices](https://docs.docker.com/engine/security/)
- [12-Factor App](https://12factor.net/)
