#!/bin/bash

#
# Ansible Playbook Generator
# Interactive script to create Ansible playbooks from templates
#

set -e

GREEN='\033[0;32m'
BLUE='\033[0;34m'
YELLOW='\033[1;33m'
NC='\033[0m'

echo -e "${BLUE}================================${NC}"
echo -e "${BLUE}Ansible Playbook Generator${NC}"
echo -e "${BLUE}================================${NC}"
echo ""

# Get project name
read -p "Enter playbook name (e.g., prod-servers): " PLAYBOOK_NAME
if [ -z "$PLAYBOOK_NAME" ]; then
    echo -e "${YELLOW}Error: Playbook name is required${NC}"
    exit 1
fi

# Check if directory exists
if [ -d "$PLAYBOOK_NAME" ]; then
    echo -e "${YELLOW}Warning: Directory $PLAYBOOK_NAME already exists${NC}"
    read -p "Overwrite? (y/N): " OVERWRITE
    if [ "$OVERWRITE" != "y" ] && [ "$OVERWRITE" != "Y" ]; then
        echo "Aborted"
        exit 0
    fi
    rm -rf "$PLAYBOOK_NAME"
fi

# Select template
echo ""
echo "Select playbook template:"
echo "1) server-provisioning  - Complete server setup"
echo "2) web-server-deploy    - Web application deployment"
echo "3) database-setup       - Database installation (coming soon)"
echo "4) security-hardening   - Security baseline configuration"
echo "5) monitoring-setup     - Monitoring agents (coming soon)"
echo "6) backup-automation    - Automated backups (coming soon)"
echo ""
read -p "Enter choice (1-6): " TEMPLATE_CHOICE

case $TEMPLATE_CHOICE in
    1)
        TEMPLATE="server-provisioning"
        ;;
    2)
        TEMPLATE="web-server-deploy"
        ;;
    4)
        TEMPLATE="security-hardening"
        ;;
    3|5|6)
        echo -e "${YELLOW}Template coming soon. Using server-provisioning instead.${NC}"
        TEMPLATE="server-provisioning"
        ;;
    *)
        echo -e "${YELLOW}Invalid choice. Using server-provisioning${NC}"
        TEMPLATE="server-provisioning"
        ;;
esac

# Get script directory
SCRIPT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
TEMPLATE_DIR="$SCRIPT_DIR/../templates/$TEMPLATE"

if [ ! -d "$TEMPLATE_DIR" ]; then
    echo -e "${YELLOW}Error: Template directory not found: $TEMPLATE_DIR${NC}"
    exit 1
fi

# Copy template
echo ""
echo -e "${GREEN}Creating playbook from template: $TEMPLATE${NC}"
cp -r "$TEMPLATE_DIR" "$PLAYBOOK_NAME"

# Create ansible.cfg
cat > "$PLAYBOOK_NAME/ansible.cfg" << 'EOF'
[defaults]
inventory = inventory/hosts
host_key_checking = False
retry_files_enabled = False
gathering = smart
fact_caching = jsonfile
fact_caching_connection = /tmp/ansible_facts
fact_caching_timeout = 3600

[privilege_escalation]
become = True
become_method = sudo
become_user = root
become_ask_pass = False
EOF

echo -e "${GREEN}Created ansible.cfg${NC}"

# Create .gitignore
cat > "$PLAYBOOK_NAME/.gitignore" << 'EOF'
*.retry
*.pyc
__pycache__/
.vault_pass
secrets.yml
*.log
/tmp/
EOF

echo -e "${GREEN}Created .gitignore${NC}"

echo ""
echo -e "${GREEN}================================${NC}"
echo -e "${GREEN}Playbook created successfully!${NC}"
echo -e "${GREEN}================================${NC}"
echo ""
echo "Next steps:"
echo "1. cd $PLAYBOOK_NAME"
echo "2. Edit inventory/hosts with your servers"
echo "3. Customize variables in playbook.yml"
echo "4. Test: ansible-playbook playbook.yml --check"
echo "5. Run: ansible-playbook playbook.yml"
echo ""
echo "Validation:"
echo "  ansible-lint playbook.yml"
echo "  ansible-playbook --syntax-check playbook.yml"
echo ""
