#!/bin/bash

#
# Ansible Playbook Test Suite
# Validates all Ansible playbook templates
#

set -e

SCRIPT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
TEMPLATES_DIR="$SCRIPT_DIR/../templates"

echo "Testing Ansible Playbook Templates"
echo "==================================="
echo ""

TESTS_PASSED=0
TESTS_FAILED=0

# Check if ansible is installed
if ! command -v ansible &> /dev/null; then
    echo "⚠️  Ansible not installed - skipping syntax checks"
    echo "   Install: pip install ansible"
    SKIP_ANSIBLE=true
fi

# Test each template
for template_dir in "$TEMPLATES_DIR"/*; do
    if [ ! -d "$template_dir" ]; then
        continue
    fi

    template_name=$(basename "$template_dir")
    echo "Testing template: $template_name"

    # Find playbook files
    playbooks=$(find "$template_dir" -name "*.yml" -o -name "*.yaml" | grep -v "inventory")

    if [ -z "$playbooks" ]; then
        echo "  ⚠️  No playbooks found"
        continue
    fi

    for playbook in $playbooks; do
        playbook_name=$(basename "$playbook")

        # Test YAML syntax with Python
        if python3 -c "import yaml; yaml.safe_load(open('$playbook'))" 2>/dev/null; then
            echo "  ✅ $playbook_name - Valid YAML"
        else
            echo "  ❌ $playbook_name - Invalid YAML"
            TESTS_FAILED=$((TESTS_FAILED + 1))
            continue
        fi

        # Test Ansible syntax if available
        if [ "$SKIP_ANSIBLE" != "true" ]; then
            if ansible-playbook "$playbook" --syntax-check 2>/dev/null; then
                echo "  ✅ $playbook_name - Valid Ansible syntax"
                TESTS_PASSED=$((TESTS_PASSED + 1))
            else
                echo "  ❌ $playbook_name - Ansible syntax errors"
                TESTS_FAILED=$((TESTS_FAILED + 1))
            fi
        fi
    done

    # Check for required files
    if [ -f "$template_dir/README.md" ]; then
        echo "  ✅ README.md present"
    else
        echo "  ⚠️  No README.md"
    fi

    echo ""
done

# Summary
echo "==================================="
echo "Test Summary:"
echo "  Passed: $TESTS_PASSED"
echo "  Failed: $TESTS_FAILED"
echo ""

if [ $TESTS_FAILED -eq 0 ]; then
    echo "✅ All playbook tests passed!"
    exit 0
else
    echo "❌ Some tests failed"
    exit 1
fi
