# Ansible Best Practices (2025)

Modern best practices for Ansible 2.16+ following industry standards and security guidelines.

## Project Structure

### Recommended Directory Layout

```
ansible-project/
├── inventory/
│   ├── production/
│   │   ├── hosts
│   │   └── group_vars/
│   └── staging/
│       ├── hosts
│       └── group_vars/
├── roles/
│   ├── common/
│   ├── webserver/
│   └── database/
├── playbooks/
│   ├── site.yml
│   ├── webservers.yml
│   └── databases.yml
├── group_vars/
│   └── all.yml
├── host_vars/
├── ansible.cfg
└── README.md
```

## Core Principles

### 1. Idempotency

Playbooks should be safe to run multiple times:

```yaml
# ✅ Good - Idempotent
- name: Ensure nginx is installed
  apt:
    name: nginx
    state: present

# ❌ Bad - Not idempotent
- name: Install nginx
  shell: apt-get install nginx
```

### 2. Use Roles for Modularity

Break complex playbooks into reusable roles:

```yaml
- hosts: webservers
  roles:
    - common
    - nginx
    - application
```

### 3. Variable Management

**Precedence order (low to high):**
1. role defaults
2. inventory file/group_vars
3. playbook vars
4. extra vars (command line)

**Best practices:**
- Use `group_vars/all.yml` for common variables
- Use `host_vars/` for host-specific values
- Use `defaults/` in roles for sensible defaults
- Use `vars/` in roles for role-specific variables

### 4. Dynamic Inventories

For cloud environments, use dynamic inventories:

```yaml
# aws_ec2.yml
plugin: aws_ec2
regions:
  - us-east-1
filters:
  tag:Environment: production
keyed_groups:
  - key: tags.Role
    prefix: role
```

## Security Best Practices

### 1. Ansible Vault for Secrets

Encrypt sensitive data:

```bash
# Create encrypted file
ansible-vault create secrets.yml

# Edit encrypted file
ansible-vault edit secrets.yml

# Use in playbook
ansible-playbook playbook.yml --ask-vault-pass
```

**Never commit unencrypted secrets to version control!**

### 2. Use become Judiciously

```yaml
# ✅ Good - Only escalate when needed
- name: Install package
  apt:
    name: nginx
  become: yes

- name: Read config
  slurp:
    src: /etc/nginx/nginx.conf
  # No become needed for reading
```

### 3. SSH Key Authentication

**Disable password authentication:**

```yaml
- name: Disable password auth
  lineinfile:
    path: /etc/ssh/sshd_config
    regexp: '^PasswordAuthentication'
    line: 'PasswordAuthentication no'
```

## Performance Optimization

### 1. Fact Caching

Enable fact caching to speed up subsequent runs:

```ini
# ansible.cfg
[defaults]
gathering = smart
fact_caching = jsonfile
fact_caching_connection = /tmp/ansible_facts
fact_caching_timeout = 3600
```

### 2. Parallel Execution

Increase parallelism:

```ini
# ansible.cfg
[defaults]
forks = 20
```

Or per playbook:

```yaml
- hosts: all
  serial: 10  # Run on 10 hosts at a time
```

### 3. Pipelining

Enable SSH pipelining for faster execution:

```ini
# ansible.cfg
[ssh_connection]
pipelining = True
```

### 4. Async Tasks

For long-running tasks:

```yaml
- name: Long running task
  command: /usr/bin/long_task
  async: 3600
  poll: 0
  register: long_task

- name: Check task status
  async_status:
    jid: "{{ long_task.ansible_job_id }}"
  register: job_result
  until: job_result.finished
  retries: 30
  delay: 10
```

## Playbook Best Practices

### 1. Use Handlers

Handlers run only when notified and only once:

```yaml
tasks:
  - name: Update nginx config
    template:
      src: nginx.conf.j2
      dest: /etc/nginx/nginx.conf
    notify: Restart nginx

handlers:
  - name: Restart nginx
    service:
      name: nginx
      state: restarted
```

### 2. Tags for Selective Execution

```yaml
- name: Install packages
  apt:
    name: "{{ item }}"
  loop: "{{ packages }}"
  tags:
    - packages
    - install

# Run only tagged tasks
# ansible-playbook playbook.yml --tags install
```

### 3. Check Mode (Dry Run)

Test changes before applying:

```bash
ansible-playbook playbook.yml --check
```

Make tasks support check mode:

```yaml
- name: Task that supports check mode
  command: /usr/bin/some_command
  check_mode: no  # Always run, even in check mode
```

### 4. Error Handling

```yaml
- name: Task that might fail
  command: /usr/bin/might_fail
  register: result
  failed_when: result.rc != 0 and result.rc != 2
  ignore_errors: yes

- name: Run only if previous task failed
  debug:
    msg: "Previous task failed"
  when: result is failed
```

### 5. Blocks for Grouping

```yaml
- block:
    - name: Task 1
      command: /bin/task1
    - name: Task 2
      command: /bin/task2
  rescue:
    - name: Handle error
      debug:
        msg: "Something failed"
  always:
    - name: Always run
      debug:
        msg: "Cleanup"
```

## Role Best Practices

### 1. Role Directory Structure

```
roles/webserver/
├── defaults/
│   └── main.yml       # Default variables
├── files/             # Static files
├── handlers/
│   └── main.yml       # Handlers
├── meta/
│   └── main.yml       # Role dependencies
├── tasks/
│   └── main.yml       # Main task list
├── templates/         # Jinja2 templates
│   └── config.j2
├── vars/
│   └── main.yml       # Role variables
└── README.md
```

### 2. Role Dependencies

In `meta/main.yml`:

```yaml
dependencies:
  - role: common
  - role: nginx
    vars:
      nginx_port: 8080
```

### 3. Galaxy Roles

Use Ansible Galaxy for community roles:

```bash
# Install role
ansible-galaxy install geerlingguy.nginx

# Requirements file
# requirements.yml
roles:
  - name: geerlingguy.nginx
    version: 3.1.4
```

## Testing

### 1. Ansible-Lint

Validate playbooks against best practices:

```bash
ansible-lint playbook.yml
```

### 2. Molecule

Test roles in isolated environments:

```bash
molecule init role my-role
molecule test
```

### 3. Syntax Check

```bash
ansible-playbook playbook.yml --syntax-check
```

## CI/CD Integration

### GitLab CI Example

```yaml
test:
  script:
    - ansible-lint playbooks/
    - ansible-playbook playbooks/site.yml --syntax-check

deploy:
  script:
    - ansible-playbook -i inventory/production playbooks/site.yml
  only:
    - main
```

## Documentation

### 1. Playbook Comments

```yaml
---
# Playbook: Web Server Deployment
# Purpose: Deploy web application with Nginx
# Author: DevOps Team
# Updated: 2025-01-15

- name: Deploy web application
  hosts: webservers
  # ... tasks
```

### 2. Variable Documentation

```yaml
# vars/main.yml
---
# Web server configuration
nginx_port: 80  # Port for Nginx to listen on
nginx_worker_processes: auto  # Number of worker processes
```

### 3. README for Roles

Include comprehensive README.md in each role explaining:
- Purpose
- Requirements
- Variables
- Dependencies
- Example playbook
- License

## Common Patterns

### 1. Conditional Execution

```yaml
- name: Install on Ubuntu
  apt:
    name: nginx
  when: ansible_distribution == "Ubuntu"

- name: Install on CentOS
  yum:
    name: nginx
  when: ansible_os_family == "RedHat"
```

### 2. Loops

```yaml
# Simple loop
- name: Create users
  user:
    name: "{{ item }}"
  loop:
    - alice
    - bob
    - charlie

# Loop with dictionaries
- name: Create users with details
  user:
    name: "{{ item.name }}"
    groups: "{{ item.groups }}"
  loop:
    - { name: 'alice', groups: 'sudo' }
    - { name: 'bob', groups: 'users' }
```

### 3. Templates

Use Jinja2 templates for dynamic configurations:

```jinja2
{# templates/nginx.conf.j2 #}
server {
    listen {{ nginx_port }};
    server_name {{ server_name }};

    {% for location in locations %}
    location {{ location.path }} {
        proxy_pass {{ location.backend }};
    }
    {% endfor %}
}
```

## Troubleshooting

### 1. Verbose Mode

```bash
# -v: verbose
# -vv: more verbose
# -vvv: connection debugging
# -vvvv: more connection debugging
ansible-playbook playbook.yml -vvv
```

### 2. Debug Module

```yaml
- name: Show variable
  debug:
    var: my_variable

- name: Show message
  debug:
    msg: "Value is {{ my_variable }}"
```

### 3. Step Mode

Run one task at a time:

```bash
ansible-playbook playbook.yml --step
```

## Version Control

### .gitignore

```gitignore
*.retry
*.pyc
__pycache__/
.vault_pass
*.log
/tmp/
.ansible/
```

### Commit Messages

Follow conventional commits:
```
feat(webserver): add nginx SSL support
fix(database): correct PostgreSQL connection settings
docs(readme): update installation instructions
```

## Summary Checklist

- [ ] Use roles for modularity
- [ ] Implement idempotent tasks
- [ ] Encrypt secrets with ansible-vault
- [ ] Use dynamic inventories for cloud
- [ ] Enable fact caching
- [ ] Add tags for selective execution
- [ ] Implement error handling
- [ ] Use handlers for service restarts
- [ ] Write comprehensive documentation
- [ ] Test with ansible-lint and check mode
- [ ] Use version control
- [ ] Implement CI/CD testing
- [ ] Follow naming conventions
- [ ] Keep playbooks DRY (Don't Repeat Yourself)
- [ ] Regular security audits

## Resources

- [Ansible Documentation](https://docs.ansible.com/)
- [Ansible Best Practices](https://docs.ansible.com/ansible/latest/user_guide/playbooks_best_practices.html)
- [Ansible Galaxy](https://galaxy.ansible.com/)
- [Ansible Lint Rules](https://ansible-lint.readthedocs.io/)
- [Molecule Testing](https://molecule.readthedocs.io/)
